#!/bin/bash

# gemlogtxt2html3.sh
# Uso: gemlogtxt2html3.sh archivo_entradas.gemlog

ARCHIVO_ENTRADAS="$1"
DIRECTORIO_SALIDA="html3"
DIRECTORIO_TAGS="$DIRECTORIO_SALIDA/tags"

# =============================================
# CONFIGURACIÓN PERSONALIZABLE DEL BLOG HTML3
# =============================================

# Textos del índice
TITULO_INDICE="Blog de _-~Caleb~-_ 📝"
BIENVENIDA_INDICE="Este es el blog del sitio, espero disfrutes de él:"

# Footer personalizado
TEXTO_FOOTER="Este blog fue generado automáticamente con gemlogtxt2html3.sh"

# =============================================
# FIN DE CONFIGURACIÓN
# =============================================

# Verificar que se proporcionó un archivo
if [ -z "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: Debes especificar un archivo de entradas"
    echo "Uso: $0 archivo_entradas.gemlog"
    exit 1
fi

# Verificar que el archivo existe
if [ ! -f "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: El archivo '$ARCHIVO_ENTRADAS' no existe"
    exit 1
fi

# Crear directorios de salida si no existen
mkdir -p "$DIRECTORIO_SALIDA"
mkdir -p "$DIRECTORIO_TAGS"

# Array para almacenar información del índice
declare -A etiquetas  # Para el índice de tags
declare -a entradas_index
declare -a entradas_programadas
declare -a borradores  # NUEVO: Array para borradores

# Función para obtener la fecha de hoy en formato AAAA-MM-DD
obtener_fecha_hoy() {
    date +%Y-%m-%d
}

# Función para validar fecha
validar_fecha() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    
    # Verificar formato AAAA-MM-DD
    if [[ ! "$fecha" =~ ^[0-9]{4}-[0-9]{2}-[0-9]{2}$ ]]; then
        echo "❌ ERROR: Formato de fecha inválido '$fecha' en entrada '$titulo'"
        echo "   El formato debe ser AAAA-MM-DD (ej: 2024-01-15)"
        return 1
    fi
    
    # Verificar que la fecha es válida usando date
    if ! date -d "$fecha" >/dev/null 2>&1; then
        echo "❌ ERROR: Fecha inválida '$fecha' en entrada '$titulo'"
        echo "   La fecha no existe en el calendario"
        return 1
    fi
    
    return 0
}

# Función para comparar fechas (0 si fecha1 <= fecha2, 1 si no)
fecha_menor_o_igual() {
    local fecha1="$1"
    local fecha2="$2"
    
    # Convertir a timestamp para comparación numérica
    local ts1=$(date -d "$fecha1" +%s 2>/dev/null)
    local ts2=$(date -d "$fecha2" +%s 2>/dev/null)
    
    if [[ $ts1 -le $ts2 ]]; then
        return 0  # verdadero
    else
        return 1  # falso
    fi
}

# Función para procesar línea de texto (conversión básica) - CORREGIDA
procesar_linea() {
    local linea="$1"
    
    # Si es una línea vacía, mantenerla
    if [[ -z "$linea" ]]; then
        echo "<p></p>"
        return
    fi
    
    # Si es un bloque de código (ya manejado separadamente)
    if [[ "$linea" =~ ^\`\`\` ]]; then
        return
    fi
    
    # Si es un encabezado
    if [[ "$linea" =~ ^#+\ + ]]; then
        local nivel=$(echo "$linea" | grep -o '^#\+' | wc -c)
        local contenido=$(echo "$linea" | sed 's/^#\+\ //')
        case $nivel in
            2) echo "<h1>$contenido</h1>" ;;
            3) echo "<h2>$contenido</h2>" ;;
            4) echo "<h3>$contenido</h3>" ;;
            *) echo "<h1>$contenido</h1>" ;;
        esac
        return
    fi
    
    # Si es una lista
    if [[ "$linea" =~ ^\*\ + ]]; then
        local contenido=$(echo "$linea" | sed 's/^\*\ //')
        echo "<li>$contenido</li>"
        return
    fi
    
    # Texto normal
    echo "<p>$linea</p>"
}

# Función para procesar enlaces
procesar_enlace() {
    local url="$1"
    local texto="$2"
    
    # Detectar imágenes
    if [[ "$url" =~ \.(jpg|jpeg|png|gif|svg|bmp)$ ]]; then
        echo "<p><img src=\"$url\" alt=\"$texto\" border=\"0\"></p>"
        return
    fi
    
    # Detectar protocolos no HTTP
    local badge=""
    case "$url" in
        gemini://*) badge="[gemini]" ;;
        gopher://*) badge="[gopher]" ;;
        spartan://*) badge="[spartan]" ;;
        finger://*) badge="[finger]" ;;
        telnet://*) badge="[telnet]" ;;
        ftp://*) badge="[ftp]" ;;
        *) badge="" ;;
    esac
    
    if [[ -n "$badge" ]]; then
        echo "<p><strong>$badge</strong> <a href=\"$url\">$texto</a></p>"
    else
        echo "<p><a href=\"$url\">$texto</a></p>"
    fi
}

# Función para procesar etiquetas
procesar_etiquetas() {
    local tags_linea="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local fecha="$4"
    
    # Extraer tags (palabras que empiezan con #)
    while [[ $tags_linea =~ (#[[:alnum:]]+) ]]; do
        local tag="${BASH_REMATCH[1]}"
        tags_linea="${tags_linea#*${BASH_REMATCH[1]}}"
        
        # Agregar entrada a este tag
        if [[ -z "${etiquetas[$tag]}" ]]; then
            etiquetas[$tag]="$fecha|$nombre_archivo|$titulo"
        else
            etiquetas[$tag]="${etiquetas[$tag]}"$'\n'"$fecha|$nombre_archivo|$titulo"
        fi
    done
}

# Función para crear archivo individual de etiqueta
crear_archivo_etiqueta() {
    local tag="$1"
    local entradas="$2"
    
    # Quitar el # del nombre del archivo
    local nombre_tag=$(echo "$tag" | sed 's/^#//')
    local archivo_tag="$DIRECTORIO_TAGS/${nombre_tag}.html"
    
    cat > "$archivo_tag" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>$tag - $TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="../styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <CENTER><H1>$tag</H1></CENTER>
    
    <DIV ALIGN="center">
EOF

    # Ordenar entradas por fecha (más reciente primero)
    printf "%s\n" "$entradas" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        archivo_sin_ext="${nombre_archivo%.*}"
        echo "        <P><A HREF=\"../${archivo_sin_ext}.html\"><STRONG>$titulo</STRONG></A><BR>" >> "$archivo_tag"
        echo "        <FONT SIZE=\"-1\" COLOR=\"#666666\">$fecha</FONT></P>" >> "$archivo_tag"
    done

    cat >> "$archivo_tag" << EOF
    </DIV>
    
    <HR>
    <P ALIGN="center">
    <A HREF="../index.html">Volver al índice principal</A> | 
    <A HREF="../tags.html">Ver todas las etiquetas</A>
    </P>
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

    echo "Creado: $archivo_tag"
}

# Función para procesar cada entrada - CORREGIDA
procesar_entrada() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local contenido="$4"
    local tiene_tags=false
    local tags_linea=""
    
    local fecha_hoy=$(obtener_fecha_hoy)
    
    echo "Procesando entrada: $fecha, $nombre_archivo, '$titulo'"
    
    # NUEVO: Verificar si es un borrador (case-insensitive)
    if [[ "${fecha,,}" == "draft" ]]; then
        borradores+=("$nombre_archivo")
        echo "📝 BORRADOR: '$titulo' - No se publicará en el índice"
        
        # Asegurar que el nombre del archivo tenga extensión .html
        local nombre_sin_ext="${nombre_archivo%.*}"
        nombre_archivo="${nombre_sin_ext}.html"
        
        # Buscar línea de tags (primera línea que empieza con #)
        local temp_file=$(mktemp)
        echo "$contenido" > "$temp_file"
        
        while IFS= read -r linea; do
            if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
                tags_linea="$linea"
                tiene_tags=true
                # Eliminar solo la línea exacta de tags usando grep -v
                contenido=$(grep -vFx "$linea" "$temp_file")
                break
            fi
        done < "$temp_file"
        rm -f "$temp_file"
        
        # Ruta completa del archivo
        local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
        
        # Crear el archivo HTML3 para borrador
        cat > "$archivo_salida" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>$titulo - $TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <P ALIGN="right"><FONT SIZE="-1"><STRONG>Estado:</STRONG> Borrador</FONT></P>
    <CENTER><H1>$titulo</H1></CENTER>
EOF

        # Añadir tags si existen
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            cat >> "$archivo_salida" << EOF
    <P><STRONG>Etiquetas:</STRONG><BR>
EOF
            # Procesar cada tag individualmente
            while read -r tag; do
                if [[ -n "$tag" ]]; then
                    echo "    <SPAN CLASS=\"tag\">$tag</SPAN> " >> "$archivo_salida"
                fi
            done <<< "$(echo "$tags_linea" | tr ' ' '\n' | grep '^#')"
            
            cat >> "$archivo_salida" << EOF
    </P>
EOF
        fi

        cat >> "$archivo_salida" << EOF
    <DIV ALIGN="left">
EOF

        # Procesar el contenido línea por línea - CORREGIDO
        local en_bloque_codigo=false
        local bloque_codigo=""
        
        while IFS= read -r linea; do
            # Manejar bloques de código
            if [[ "$linea" =~ ^\`\`\` ]]; then
                if [[ "$en_bloque_codigo" == false ]]; then
                    en_bloque_codigo=true
                    bloque_codigo=""
                else
                    echo "<PRE>$bloque_codigo</PRE>" >> "$archivo_salida"
                    en_bloque_codigo=false
                    bloque_codigo=""
                fi
                continue
            fi
            
            if [[ "$en_bloque_codigo" == true ]]; then
                bloque_codigo="${bloque_codigo}${linea}"$'\n'
                continue
            fi
            
            # Manejar enlaces
            if [[ "$linea" =~ ^=\>\ +([^ ]+)(\ +(.+))?$ ]]; then
                local url="${BASH_REMATCH[1]}"
                local texto="${BASH_REMATCH[3]:-$url}"
                procesar_enlace "$url" "$texto" >> "$archivo_salida"
                continue
            fi
            
            # CORRECCIÓN: Procesar citas directamente aquí
            if [[ "$linea" =~ ^\>\ + ]]; then
                local contenido_sin_gt=$(echo "$linea" | sed 's/^\>\ //')
                echo "<blockquote><p>$contenido_sin_gt</p></blockquote>" >> "$archivo_salida"
                continue
            fi
            
            # Procesar línea normal (sin citas, ya las procesamos arriba)
            procesar_linea "$linea" >> "$archivo_salida"
            
        done <<< "$contenido"

        cat >> "$archivo_salida" << EOF
    </DIV>
    
    <HR>
    <P ALIGN="center">
    <A HREF="index.html">Volver al índice</A> | 
    <A HREF="tags.html">Ver todas las etiquetas</A>
    </P>
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

        echo "Creado: $archivo_salida"
        return 0
    fi
    
    # Validar fecha antes de procesar (solo si no es borrador)
    if ! validar_fecha "$fecha" "$nombre_archivo" "$titulo"; then
        echo "❌ ENTRADA OMITIDA: '$titulo' - Fecha inválida"
        return 1
    fi
    
    # Asegurar que el nombre del archivo tenga extensión .html
    local nombre_sin_ext="${nombre_archivo%.*}"
    nombre_archivo="${nombre_sin_ext}.html"
    
    # Buscar línea de tags (primera línea que empieza con #)
    local temp_file=$(mktemp)
    echo "$contenido" > "$temp_file"
    
    while IFS= read -r linea; do
        if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
            tags_linea="$linea"
            tiene_tags=true
            # Eliminar solo la línea exacta de tags usando grep -v
            contenido=$(grep -vFx "$linea" "$temp_file")
            break
        fi
    done < "$temp_file"
    rm -f "$temp_file"
    
    # Ruta completa del archivo
    local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
    
    # Crear el archivo HTML3
    cat > "$archivo_salida" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>$titulo - $TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <P ALIGN="right"><FONT SIZE="-1"><STRONG>Publicado:</STRONG> $fecha</FONT></P>
    <CENTER><H1>$titulo</H1></CENTER>
EOF

    # Añadir tags si existen
    if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
        cat >> "$archivo_salida" << EOF
    <P><STRONG>Etiquetas:</STRONG><BR>
EOF
        # Procesar cada tag individualmente
        while read -r tag; do
            if [[ -n "$tag" ]]; then
                echo "    <SPAN CLASS=\"tag\">$tag</SPAN> " >> "$archivo_salida"
            fi
        done <<< "$(echo "$tags_linea" | tr ' ' '\n' | grep '^#')"
        
        cat >> "$archivo_salida" << EOF
    </P>
EOF
    fi

    cat >> "$archivo_salida" << EOF
    <DIV ALIGN="left">
EOF

    # Procesar el contenido línea por línea - CORREGIDO
    local en_bloque_codigo=false
    local bloque_codigo=""
    
    while IFS= read -r linea; do
        # Manejar bloques de código
        if [[ "$linea" =~ ^\`\`\` ]]; then
            if [[ "$en_bloque_codigo" == false ]]; then
                en_bloque_codigo=true
                bloque_codigo=""
            else
                echo "<PRE>$bloque_codigo</PRE>" >> "$archivo_salida"
                en_bloque_codigo=false
                bloque_codigo=""
            fi
            continue
        fi
        
        if [[ "$en_bloque_codigo" == true ]]; then
            bloque_codigo="${bloque_codigo}${linea}"$'\n'
            continue
        fi
        
        # Manejar enlaces
        if [[ "$linea" =~ ^=\>\ +([^ ]+)(\ +(.+))?$ ]]; then
            local url="${BASH_REMATCH[1]}"
            local texto="${BASH_REMATCH[3]:-$url}"
            procesar_enlace "$url" "$texto" >> "$archivo_salida"
            continue
        fi
        
        # CORRECCIÓN: Procesar citas directamente aquí
        if [[ "$linea" =~ ^\>\ + ]]; then
            local contenido_sin_gt=$(echo "$linea" | sed 's/^\>\ //')
            echo "<blockquote><p>$contenido_sin_gt</p></blockquote>" >> "$archivo_salida"
            continue
        fi
        
        # Procesar línea normal (sin citas, ya las procesamos arriba)
        procesar_linea "$linea" >> "$archivo_salida"
        
    done <<< "$contenido"

    cat >> "$archivo_salida" << EOF
    </DIV>
    
    <HR>
    <P ALIGN="center">
    <A HREF="index.html">Volver al índice</A> | 
    <A HREF="tags.html">Ver todas las etiquetas</A>
    </P>
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

    echo "Creado: $archivo_salida"
    
    # Solo agregar al índice y procesar tags si la fecha es hoy o pasada
    if fecha_menor_o_igual "$fecha" "$fecha_hoy"; then
        entradas_index+=("$fecha|$nombre_archivo|$titulo")
        # Procesar tags solo para entradas publicadas
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            echo "   Tags encontrados: $tags_linea"
            procesar_etiquetas "$tags_linea" "$nombre_archivo" "$titulo" "$fecha"
        fi
        echo "✅ Añadida al índice: $titulo"
    else
        entradas_programadas+=("$fecha|$nombre_archivo|$titulo")
        echo "⏰ Programada para el futuro: $titulo (visible el $fecha)"
    fi
    
    return 0
}

# Función para ordenar etiquetas por uso (de más a menos)
obtener_etiquetas_ordenadas() {
    for tag in "${!etiquetas[@]}"; do
        local count=$(echo "${etiquetas[$tag]}" | wc -l)
        echo "$count|$tag"
    done | sort -nr | cut -d'|' -f2
}

# Función para crear el índice de etiquetas
crear_indice_etiquetas() {
    local archivo_tags="$DIRECTORIO_SALIDA/tags.html"
    
    cat > "$archivo_tags" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>Índice de Etiquetas - $TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <CENTER><H1>Índice de Etiquetas</H1></CENTER>
    
    <P>Explora las entradas por etiquetas:</P>
EOF

    # Ordenar tags por uso (de más a menos)
    if [ ${#etiquetas[@]} -gt 0 ]; then
        while IFS= read -r tag; do
            # Contar número de entradas para este tag
            local count=$(echo "${etiquetas[$tag]}" | wc -l)
            local nombre_tag=$(echo "$tag" | sed 's/^#//')
            echo "    <P><A HREF=\"tags/${nombre_tag}.html\"><STRONG>$tag</STRONG></A> ($count)</P>" >> "$archivo_tags"
            
            # Crear archivo individual para esta etiqueta
            crear_archivo_etiqueta "$tag" "${etiquetas[$tag]}"
        done < <(obtener_etiquetas_ordenadas)
    else
        echo "    <P>No hay etiquetas disponibles.</P>" >> "$archivo_tags"
    fi

    cat >> "$archivo_tags" << EOF
    
    <HR>
    <P ALIGN="center">
    <A HREF="index.html">Volver al índice principal</A>
    </P>
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

    echo "Creado: $archivo_tags"
}

# Función para crear el archivo CSS compatible con HTML3 (colores anaranjados/ocre)
crear_archivo_css() {
    local archivo_css="$DIRECTORIO_SALIDA/styles.css"
    
    cat > "$archivo_css" << 'EOF'
/* styles.css - Compatible con HTML3 - Tema anaranjado/ocre */
body {
    font-family: "Courier New", Courier, monospace;
    margin: 0;
    padding: 0;
    background-color: #F5F5DC;
    color: #333333;
    font-size: 14px;
    line-height: 1.5;
    text-align: center; /* Para centrar la tabla */
}

table {
    border-collapse: collapse;
    width: 800px; /* Ancho máximo fijo */
    margin: 0 auto; /* Centrar la tabla */
}

h1 {
    color: #CC6600;
    font-size: 24px;
    margin: 10px 0;
    padding: 0;
    text-align: center;
    border-bottom: 2px solid #CC6600;
}

h2 {
    color: #CC6600;
    font-size: 20px;
    margin: 15px 0 10px 0;
    padding: 0;
}

h3 {
    color: #996633;
    font-size: 16px;
    margin: 10px 0 5px 0;
    padding: 0;
}

p {
    margin: 10px 0;
    padding: 0;
    text-align: justify;
}

pre {
    background-color: #F8F4E9;
    border: 1px solid #D9C7A7;
    padding: 10px;
    margin: 10px 0;
    font-family: "Courier New", Courier, monospace;
    font-size: 12px;
    overflow: auto;
    white-space: pre;
}

blockquote {
    border-left: 3px solid #D9C7A7;
    margin: 10px 0;
    padding-left: 15px;
    color: #666666;
    font-style: italic;
    background-color: #F8F4E9;
}

.tag {
    background-color: #CC6600;
    color: #FFFFFF;
    padding: 2px 6px;
    margin: 2px;
    font-size: 11px;
    font-weight: bold;
}

a {
    color: #CC6600;
    text-decoration: none;
}

a:hover {
    color: #FF9900;
    text-decoration: underline;
}

a:visited {
    color: #996633;
}

hr {
    border: 0;
    height: 1px;
    background-color: #D9C7A7;
    margin: 20px 0;
}

ul {
    margin: 10px 0;
    padding-left: 20px;
}

li {
    margin: 5px 0;
}

.protocol-badge {
    font-weight: bold;
    color: #FFFFFF;
    background-color: #996633;
    padding: 1px 4px;
    margin-right: 5px;
    font-size: 10px;
}

img {
    border: 1px solid #D9C7A7;
    margin: 10px 0;
    max-width: 100%;
}

.entry-item {
    background-color: #F8F4E9;
    border-left: 3px solid #CC6600;
    padding: 8px;
    margin: 5px 0;
}

.entry-date {
    color: #666666;
    font-size: 11px;
    margin-top: 2px;
}
EOF

    echo "Creado: $archivo_css"
}

# Procesar el archivo línea por línea
fecha=""
nombre_archivo=""
titulo=""
contenido=""
en_contenido=false
errores_encontrados=0

echo "Leyendo archivo: $ARCHIVO_ENTRADAS"

# Leer el archivo preservando espacios y líneas vacías
while IFS= read -r linea; do
    # Detectar fin de entrada
    if [[ "$linea" == "<END>" ]]; then
        if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
            echo "Fin de entrada encontrado, procesando..."
            if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
                ((errores_encontrados++))
            fi
            # Resetear variables
            fecha=""
            nombre_archivo=""
            titulo=""
            contenido=""
            en_contenido=false
        fi
        continue
    fi
    
    # Saltar comentarios (solo líneas que empiezan con ')
    if [[ "$linea" =~ ^\' ]]; then
        continue
    fi
    
    # Si no tenemos fecha, esta línea es la fecha
    if [[ -z "$fecha" ]]; then
        fecha=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos fecha pero no nombre de archivo, esta línea es el nombre
    if [[ -z "$nombre_archivo" ]]; then
        nombre_archivo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos nombre de archivo pero no título, esta línea es el título
    if [[ -z "$titulo" ]]; then
        titulo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        en_contenido=true
        continue
    fi
    
    # Si estamos en el contenido, PRESERVAR la línea exactamente como está
    if [[ "$en_contenido" == true ]]; then
        if [[ -z "$contenido" ]]; then
            contenido="$linea"
        else
            contenido="$contenido"$'\n'"$linea"
        fi
    fi
    
done < "$ARCHIVO_ENTRADAS"

# Procesar última entrada si no terminó con <END>
if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
    echo "Procesando última entrada..."
    if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
        ((errores_encontrados++))
    fi
fi

# Si hay errores, mostrar advertencia
if [ $errores_encontrados -gt 0 ]; then
    echo ""
    echo "⚠️  ADVERTENCIA: Se encontraron $errores_encontrados errores en las fechas"
    echo "   Algunas entradas no fueron procesadas debido a fechas inválidas"
fi

# Crear el archivo CSS
crear_archivo_css

# Crear el archivo índice principal
archivo_index="$DIRECTORIO_SALIDA/index.html"

echo "Creando índice con ${#entradas_index[@]} entradas visibles..."

cat > "$archivo_index" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>$TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <CENTER><H1>$TITULO_INDICE</H1></CENTER>
    
    <P>$BIENVENIDA_INDICE</P>
    
    <P ALIGN="center">
    <A HREF="tags.html">Explorar por etiquetas</A>
    </P>
    
    <H2>Entradas del blog</H2>
EOF

# Ordenar entradas por fecha (más reciente primero) y agregar al índice
if [ ${#entradas_index[@]} -gt 0 ]; then
    printf "%s\n" "${entradas_index[@]}" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        archivo_sin_ext="${nombre_archivo%.*}"
        echo "    <DIV CLASS=\"entry-item\">" >> "$archivo_index"
        echo "        <A HREF=\"$archivo_sin_ext.html\"><STRONG>$titulo</STRONG></A><BR>" >> "$archivo_index"
        echo "        <SPAN CLASS=\"entry-date\">$fecha</SPAN>" >> "$archivo_index"
        echo "    </DIV>" >> "$archivo_index"
    done
else
    echo "    <P>No hay entradas disponibles.</P>" >> "$archivo_index"
fi

cat >> "$archivo_index" << EOF
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

# Crear índice de etiquetas (solo con entradas publicadas)
if [ ${#etiquetas[@]} -gt 0 ]; then
    crear_indice_etiquetas
else
    # Crear archivo tags.html vacío
    cat > "$DIRECTORIO_SALIDA/tags.html" << EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 3.2 Final//EN">
<HTML>
<HEAD>
<TITLE>Índice de Etiquetas - $TITULO_INDICE</TITLE>
<LINK REL="stylesheet" HREF="styles.css" TYPE="text/css">
</HEAD>
<BODY BGCOLOR="#F5F5DC" TEXT="#333333" LINK="#CC6600" VLINK="#996633" ALINK="#FF9900">
<CENTER>
<TABLE WIDTH="800" BORDER="0" CELLSPACING="0" CELLPADDING="10">
<TR>
<TD BGCOLOR="#F8F4E9">
    <CENTER><H1>Índice de Etiquetas</H1></CENTER>
    
    <P>No hay etiquetas disponibles.</P>
    
    <HR>
    <P ALIGN="center">
    <A HREF="index.html">Volver al índice principal</A>
    </P>
    
    <HR>
    <P ALIGN="center"><SMALL>$TEXTO_FOOTER</SMALL></P>
</TD>
</TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF
fi

echo ""
echo "=== PROCESAMIENTO COMPLETADO ==="
echo "Archivos creados en: $DIRECTORIO_SALIDA/"
echo "Entradas en el índice: ${#entradas_index[@]}"
echo "Entradas programadas: ${#entradas_programadas[@]}"
echo "Borradores: ${#borradores[@]}"  # NUEVO: Mostrar borradores
echo "Etiquetas procesadas: ${#etiquetas[@]}"
echo "Errores de fecha: $errores_encontrados"
echo "Índice generado: $archivo_index"
echo "Índice de etiquetas: $DIRECTORIO_SALIDA/tags.html"
echo "Archivos de etiquetas: $DIRECTORIO_TAGS/"

# NUEVO: Mostrar resumen de borradores
if [ ${#borradores[@]} -gt 0 ]; then
    echo ""
    echo "=== BORRADORES ==="
    for borrador in "${borradores[@]}"; do
        echo "📝 $borrador"
    done
fi

# MEJORADO: Mostrar resumen de entradas programadas ORDENADO por fecha y con días restantes
if [ ${#entradas_programadas[@]} -gt 0 ]; then
    echo ""
    echo "=== ENTRADAS PROGRAMADAS ==="
    
    # Obtener fecha actual en timestamp
    fecha_hoy_ts=$(date +%s)
    
    # Ordenar entradas programadas por fecha (más cercana primero)
    printf "%s\n" "${entradas_programadas[@]}" | sort -t'|' -k1,1 | while IFS='|' read -r fecha nombre_archivo titulo; do
        # Convertir fecha programada a timestamp
        fecha_programada_ts=$(date -d "$fecha" +%s 2>/dev/null)
        
        # Calcular días restantes
        if [[ -n "$fecha_programada_ts" ]]; then
            segundos_restantes=$((fecha_programada_ts - fecha_hoy_ts))
            dias_restantes=$((segundos_restantes / 86400))
            
            if [[ $dias_restantes -eq 1 ]]; then
                texto_dias="(Dentro de 1 día)"
            elif [[ $dias_restantes -eq 0 ]]; then
                texto_dias="(Hoy)"
            else
                texto_dias="(Dentro de $dias_restantes días)"
            fi
        else
            texto_dias="(Fecha inválida)"
        fi
        
        echo "⏰ $nombre_archivo - Visible el $fecha $texto_dias"
    done
fi

# Si hubo errores, salir con código de error
if [ $errores_encontrados -gt 0 ]; then
    exit 1
fi
