#!/bin/bash

# gemlogtxt2html.sh
# Uso: gemlogtxt2html.sh archivo_entradas.gemlog

ARCHIVO_ENTRADAS="$1"
DIRECTORIO_SALIDA="html"
DIRECTORIO_TAGS="$DIRECTORIO_SALIDA/tags"

# =============================================
# CONFIGURACIÓN PERSONALIZABLE DEL BLOG HTML
# =============================================

# Textos del índice
TITULO_INDICE="Blog de _-~Caleb~-_ 📝"
BIENVENIDA_INDICE="Este es el blog del sitio, espero disfrutes de él:"

# Footer personalizado
TEXTO_FOOTER="Este blog fue generado automáticamente con gemlogtxt2html.sh"

# =============================================
# FIN DE CONFIGURACIÓN
# =============================================

# Verificar que se proporcionó un archivo
if [ -z "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: Debes especificar un archivo de entradas"
    echo "Uso: $0 archivo_entradas.gemlog"
    exit 1
fi

# Verificar que el archivo existe
if [ ! -f "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: El archivo '$ARCHIVO_ENTRADAS' no existe"
    exit 1
fi

# Crear directorios de salida si no existen
mkdir -p "$DIRECTORIO_SALIDA"
mkdir -p "$DIRECTORIO_TAGS"

# Array para almacenar información del índice
declare -A etiquetas  # Para el índice de tags
declare -a entradas_index
declare -a entradas_programadas
declare -a borradores

# Función para obtener la fecha de hoy en formato AAAA-MM-DD
obtener_fecha_hoy() {
    date +%Y-%m-%d
}

# Función para validar fecha
validar_fecha() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    
    # Verificar formato AAAA-MM-DD
    if [[ ! "$fecha" =~ ^[0-9]{4}-[0-9]{2}-[0-9]{2}$ ]]; then
        echo "❌ ERROR: Formato de fecha inválido '$fecha' en entrada '$titulo'"
        echo "   El formato debe ser AAAA-MM-DD (ej: 2024-01-15)"
        return 1
    fi
    
    # Verificar que la fecha es válida usando date
    if ! date -d "$fecha" >/dev/null 2>&1; then
        echo "❌ ERROR: Fecha inválida '$fecha' en entrada '$titulo'"
        echo "   La fecha no existe en el calendario"
        return 1
    fi
    
    return 0
}

# Función para comparar fechas (0 si fecha1 <= fecha2, 1 si no)
fecha_menor_o_igual() {
    local fecha1="$1"
    local fecha2="$2"
    
    # Convertir a timestamp para comparación numérica
    local ts1=$(date -d "$fecha1" +%s 2>/dev/null)
    local ts2=$(date -d "$fecha2" +%s 2>/dev/null)
    
    if [[ $ts1 -le $ts2 ]]; then
        return 0  # verdadero
    else
        return 1  # falso
    fi
}

# Función para procesar línea de texto (conversión básica)
procesar_linea() {
    local linea="$1"
    
    # Si es una línea vacía, mantenerla
    if [[ -z "$linea" ]]; then
        echo "<p></p>"
        return
    fi
    
    # Si es un bloque de código (ya manejado separadamente)
    if [[ "$linea" =~ ^\`\`\` ]]; then
        return
    fi
    
    # Si es un encabezado
    if [[ "$linea" =~ ^#+\ + ]]; then
        local nivel=$(echo "$linea" | grep -o '^#\+' | wc -c)
        local contenido=$(echo "$linea" | sed 's/^#\+\ //')
        case $nivel in
            2) echo "<h1>$contenido</h1>" ;;
            3) echo "<h2>$contenido</h2>" ;;
            4) echo "<h3>$contenido</h3>" ;;
            *) echo "<h1>$contenido</h1>" ;;
        esac
        return
    fi
    
    # Si es una lista
    if [[ "$linea" =~ ^\*\ + ]]; then
        local contenido=$(echo "$linea" | sed 's/^\*\ //')
        echo "<li>$contenido</li>"
        return
    fi
    
    # Si es una cita
    if [[ "$linea" =~ ^\>\ + ]]; then
        local contenido=$(echo "$linea" | sed 's/^\>\ //')
        echo "<blockquote><p>$contenido</p></blockquote>"
        return
    fi
    
    # Texto normal
    echo "<p>$linea</p>"
}

# Función para procesar enlaces
procesar_enlace() {
    local url="$1"
    local texto="$2"
    
    # Detectar imágenes
    if [[ "$url" =~ \.(jpg|jpeg|png|gif|svg|bmp)$ ]]; then
        echo "<p><img src=\"$url\" alt=\"$texto\"></p>"
        return
    fi
    
    # Detectar protocolos no HTTP
    local badge=""
    case "$url" in
        gemini://*) badge="[gemini]" ;;
        gopher://*) badge="[gopher]" ;;
        spartan://*) badge="[spartan]" ;;
        finger://*) badge="[finger]" ;;
        telnet://*) badge="[telnet]" ;;
        ftp://*) badge="[ftp]" ;;
        *) badge="" ;;
    esac
    
    if [[ -n "$badge" ]]; then
        echo "<p><strong>$badge</strong> <a href=\"$url\">$texto</a></p>"
    else
        echo "<p><a href=\"$url\">$texto</a></p>"
    fi
}

# Función para procesar etiquetas
procesar_etiquetas() {
    local tags_linea="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local fecha="$4"
    
    # Extraer tags (palabras que empiezan con #)
    while [[ $tags_linea =~ (#[[:alnum:]]+) ]]; do
        local tag="${BASH_REMATCH[1]}"
        tags_linea="${tags_linea#*${BASH_REMATCH[1]}}"
        
        # Agregar entrada a este tag
        if [[ -z "${etiquetas[$tag]}" ]]; then
            etiquetas[$tag]="$fecha|$nombre_archivo|$titulo"
        else
            etiquetas[$tag]="${etiquetas[$tag]}"$'\n'"$fecha|$nombre_archivo|$titulo"
        fi
    done
}

# Función para crear archivo individual de etiqueta
crear_archivo_etiqueta() {
    local tag="$1"
    local entradas="$2"
    
    # Quitar el # del nombre del archivo
    local nombre_tag=$(echo "$tag" | sed 's/^#//')
    local archivo_tag="$DIRECTORIO_TAGS/${nombre_tag}.html"
    
    cat > "$archivo_tag" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>$tag - $TITULO_INDICE</title>
    <link rel="stylesheet" href="../styles.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>$tag</h1>
        </header>

        <main>
EOF

    # Ordenar entradas por fecha (más reciente primero)
    printf "%s\n" "$entradas" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        archivo_sin_ext="${nombre_archivo%.*}"
        echo "            <div class=\"entry-item\">" >> "$archivo_tag"
        echo "                <a href=\"../${archivo_sin_ext}.html\">$titulo</a>" >> "$archivo_tag"
        echo "                <div class=\"entry-date\">$fecha</div>" >> "$archivo_tag"
        echo "            </div>" >> "$archivo_tag"
    done

    cat >> "$archivo_tag" << EOF
        </main>

        <div class="nav">
            <a href="../index.html">Volver al índice principal</a>
            <a href="../tags.html">Ver todas las etiquetas</a>
        </div>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF

    echo "Creado: $archivo_tag"
}

# Función para procesar cada entrada
procesar_entrada() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local contenido="$4"
    local tiene_tags=false
    local tags_linea=""
    
    local fecha_hoy=$(obtener_fecha_hoy)
    
    echo "Procesando entrada: $fecha, $nombre_archivo, '$titulo'"
    
    # Verificar si es un borrador (case-insensitive)
    if [[ "${fecha,,}" == "draft" ]]; then
        borradores+=("$nombre_archivo")
        echo "📝 BORRADOR: '$titulo' - No se publicará en el índice"
        
        # Asegurar que el nombre del archivo tenga extensión .html
        local nombre_sin_ext="${nombre_archivo%.*}"
        nombre_archivo="${nombre_sin_ext}.html"
        
        # Buscar línea de tags (primera línea que empieza con #)
        local temp_file=$(mktemp)
        echo "$contenido" > "$temp_file"
        
        while IFS= read -r linea; do
            if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
                tags_linea="$linea"
                tiene_tags=true
                # Eliminar solo la línea exacta de tags usando grep -v
                contenido=$(grep -vFx "$linea" "$temp_file")
                break
            fi
        done < "$temp_file"
        rm -f "$temp_file"
        
        # Ruta completa del archivo
        local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
        
        # Crear el archivo HTML
        cat > "$archivo_salida" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>$titulo - $TITULO_INDICE</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container">
        <header>
            <div class="meta">
                <strong>Estado:</strong> Borrador
            </div>
            <h1>$titulo</h1>
        </header>
EOF

        # Añadir tags si existen
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            cat >> "$archivo_salida" << EOF
        <div class="tags">
            <strong>Etiquetas:</strong>
EOF
            # Procesar cada tag individualmente
            while read -r tag; do
                if [[ -n "$tag" ]]; then
                    local nombre_tag=$(echo "$tag" | sed 's/^#//')
                    echo "                <span class=\"tag\">$tag</span>" >> "$archivo_salida"
                fi
            done <<< "$(echo "$tags_linea" | tr ' ' '\n' | grep '^#')"
            
            cat >> "$archivo_salida" << EOF
        </div>
EOF
        fi

        cat >> "$archivo_salida" << EOF
        <article>
EOF

        # Procesar el contenido línea por línea
        local en_bloque_codigo=false
        local bloque_codigo=""
        
        while IFS= read -r linea; do
            # Manejar bloques de código
            if [[ "$linea" =~ ^\`\`\` ]]; then
                if [[ "$en_bloque_codigo" == false ]]; then
                    en_bloque_codigo=true
                    bloque_codigo=""
                else
                    echo "<pre>$bloque_codigo</pre>" >> "$archivo_salida"
                    en_bloque_codigo=false
                    bloque_codigo=""
                fi
                continue
            fi
            
            if [[ "$en_bloque_codigo" == true ]]; then
                bloque_codigo="${bloque_codigo}${linea}"$'\n'
                continue
            fi
            
            # Manejar enlaces
            if [[ "$linea" =~ ^=\>\ +([^ ]+)(\ +(.+))?$ ]]; then
                local url="${BASH_REMATCH[1]}"
                local texto="${BASH_REMATCH[3]:-$url}"
                procesar_enlace "$url" "$texto" >> "$archivo_salida"
                continue
            fi
            
            # Procesar línea normal
            procesar_linea "$linea" >> "$archivo_salida"
            
        done <<< "$contenido"

        cat >> "$archivo_salida" << EOF
        </article>

        <div class="nav">
            <a href="index.html">Volver al índice</a>
            <a href="tags.html">Ver todas las etiquetas</a>
        </div>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF

        echo "Creado: $archivo_salida"
        return 0
    fi
    
    # Validar fecha antes de procesar (solo si no es borrador)
    if ! validar_fecha "$fecha" "$nombre_archivo" "$titulo"; then
        echo "❌ ENTRADA OMITIDA: '$titulo' - Fecha inválida"
        return 1
    fi
    
    # Asegurar que el nombre del archivo tenga extensión .html
    local nombre_sin_ext="${nombre_archivo%.*}"
    nombre_archivo="${nombre_sin_ext}.html"
    
    # Buscar línea de tags (primera línea que empieza con #)
    local temp_file=$(mktemp)
    echo "$contenido" > "$temp_file"
    
    while IFS= read -r linea; do
        if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
            tags_linea="$linea"
            tiene_tags=true
            # Eliminar solo la línea exacta de tags usando grep -v
            contenido=$(grep -vFx "$linea" "$temp_file")
            break
        fi
    done < "$temp_file"
    rm -f "$temp_file"
    
    # Ruta completa del archivo
    local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
    
    # Crear el archivo HTML
    cat > "$archivo_salida" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>$titulo - $TITULO_INDICE</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container">
        <header>
            <div class="meta">
                <strong>Publicado:</strong> $fecha
            </div>
            <h1>$titulo</h1>
        </header>
EOF

    # Añadir tags si existen
    if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
        cat >> "$archivo_salida" << EOF
        <div class="tags">
            <strong>Etiquetas:</strong>
EOF
        # Procesar cada tag individualmente
        while read -r tag; do
            if [[ -n "$tag" ]]; then
                local nombre_tag=$(echo "$tag" | sed 's/^#//')
                echo "                <span class=\"tag\">$tag</span>" >> "$archivo_salida"
            fi
        done <<< "$(echo "$tags_linea" | tr ' ' '\n' | grep '^#')"
        
        cat >> "$archivo_salida" << EOF
        </div>
EOF
    fi

    cat >> "$archivo_salida" << EOF
        <article>
EOF

    # Procesar el contenido línea por línea
    local en_bloque_codigo=false
    local bloque_codigo=""
    
    while IFS= read -r linea; do
        # Manejar bloques de código
        if [[ "$linea" =~ ^\`\`\` ]]; then
            if [[ "$en_bloque_codigo" == false ]]; then
                en_bloque_codigo=true
                bloque_codigo=""
            else
                echo "<pre>$bloque_codigo</pre>" >> "$archivo_salida"
                en_bloque_codigo=false
                bloque_codigo=""
            fi
            continue
        fi
        
        if [[ "$en_bloque_codigo" == true ]]; then
            bloque_codigo="${bloque_codigo}${linea}"$'\n'
            continue
        fi
        
        # Manejar enlaces
        if [[ "$linea" =~ ^=\>\ +([^ ]+)(\ +(.+))?$ ]]; then
            local url="${BASH_REMATCH[1]}"
            local texto="${BASH_REMATCH[3]:-$url}"
            procesar_enlace "$url" "$texto" >> "$archivo_salida"
            continue
        fi
        
        # Procesar línea normal
        procesar_linea "$linea" >> "$archivo_salida"
        
    done <<< "$contenido"

    cat >> "$archivo_salida" << EOF
        </article>

        <div class="nav">
            <a href="index.html">Volver al índice</a>
            <a href="tags.html">Ver todas las etiquetas</a>
        </div>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF

    echo "Creado: $archivo_salida"
    
    # Solo agregar al índice y procesar tags si la fecha es hoy o pasada
    if fecha_menor_o_igual "$fecha" "$fecha_hoy"; then
        entradas_index+=("$fecha|$nombre_archivo|$titulo")
        # Procesar tags solo para entradas publicadas
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            echo "   Tags encontrados: $tags_linea"
            procesar_etiquetas "$tags_linea" "$nombre_archivo" "$titulo" "$fecha"
        fi
        echo "✅ Añadida al índice: $titulo"
    else
        entradas_programadas+=("$fecha|$nombre_archivo|$titulo")
        echo "⏰ Programada para el futuro: $titulo (visible el $fecha)"
    fi
    
    return 0
}

# Función para ordenar etiquetas por uso (de más a menos)
obtener_etiquetas_ordenadas() {
    for tag in "${!etiquetas[@]}"; do
        local count=$(echo "${etiquetas[$tag]}" | wc -l)
        echo "$count|$tag"
    done | sort -nr | cut -d'|' -f2
}

# Función para crear el índice de etiquetas
crear_indice_etiquetas() {
    local archivo_tags="$DIRECTORIO_SALIDA/tags.html"
    
    cat > "$archivo_tags" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Índice de Etiquetas - $TITULO_INDICE</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>Índice de Etiquetas</h1>
        </header>

        <main>
            <p>Explora las entradas por etiquetas:</p>
EOF

    # Ordenar tags por uso (de más a menos)
    if [ ${#etiquetas[@]} -gt 0 ]; then
        while IFS= read -r tag; do
            # Contar número de entradas para este tag
            local count=$(echo "${etiquetas[$tag]}" | wc -l)
            local nombre_tag=$(echo "$tag" | sed 's/^#//')
            echo "            <div class=\"entry-item\">" >> "$archivo_tags"
            echo "                <a href=\"tags/${nombre_tag}.html\">$tag ($count)</a>" >> "$archivo_tags"
            echo "            </div>" >> "$archivo_tags"
            
            # Crear archivo individual para esta etiqueta
            crear_archivo_etiqueta "$tag" "${etiquetas[$tag]}"
        done < <(obtener_etiquetas_ordenadas)
    else
        echo "            <p>No hay etiquetas disponibles.</p>" >> "$archivo_tags"
    fi

    cat >> "$archivo_tags" << EOF
        </main>

        <div class="nav">
            <a href="index.html">Volver al índice principal</a>
        </div>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF

    echo "Creado: $archivo_tags"
}

# Función para crear el archivo CSS
crear_archivo_css() {
    local archivo_css="$DIRECTORIO_SALIDA/styles.css"
    
    cat > "$archivo_css" << 'EOF'
/* styles.css - Modo claro y oscuro automático */
:root {
    /* Colores modo claro (por defecto) */
    --bg-color: #f5f5f5;
    --container-bg: #ffffff;
    --text-color: #333333;
    --border-color: #dddddd;
    --accent-color: #3498db;
    --accent-hover: #2980b9;
    --secondary-color: #7f8c8d;
    --tag-bg: #3498db;
    --tag-color: #ffffff;
    --blockquote-border: #bdc3c7;
    --pre-bg: #f8f8f8;
    --entry-bg: #f8f9fa;
    --shadow: rgba(0,0,0,0.1);
}

/* Tema oscuro automático basado en preferencias del sistema */
@media (prefers-color-scheme: dark) {
    :root {
        --bg-color: #1a1a1a;
        --container-bg: #2d2d2d;
        --text-color: #e0e0e0;
        --border-color: #444444;
        --accent-color: #5dade2;
        --accent-hover: #3498db;
        --secondary-color: #b0b0b0;
        --tag-bg: #5dade2;
        --tag-color: #ffffff;
        --blockquote-border: #555555;
        --pre-bg: #3a3a3a;
        --entry-bg: #3a3a3a;
        --shadow: rgba(0,0,0,0.3);
    }
}

body {
    font-family: 'Courier New', monospace;
    max-width: 800px;
    margin: 0 auto;
    padding: 20px;
    line-height: 1.6;
    background-color: var(--bg-color);
    color: var(--text-color);
    transition: background-color 0.3s ease, color 0.3s ease;
}

.container {
    background: var(--container-bg);
    padding: 30px;
    border-radius: 8px;
    box-shadow: 0 2px 10px var(--shadow);
    transition: background-color 0.3s ease, box-shadow 0.3s ease;
}

/* Justificación de texto */
article p {
    text-align: justify;
    text-justify: inter-word;
}

article {
    text-align: justify;
}

/* Mantener otros elementos alineados a la izquierda */
h1, h2, h3, h4, h5, h6,
pre, code,
blockquote,
.tags,
.nav,
.footer,
header p,
.entry-item,
.entry-date {
    text-align: left;
}

h1 {
    color: var(--text-color);
    border-bottom: 2px solid var(--accent-color);
    padding-bottom: 10px;
}

h2 {
    color: var(--text-color);
    margin-top: 30px;
}

h3 {
    color: var(--secondary-color);
    margin-top: 20px;
}

pre {
    background: var(--pre-bg);
    padding: 15px;
    border-radius: 5px;
    overflow-x: auto;
    border-left: 4px solid var(--accent-color);
    font-family: 'Courier New', monospace;
    white-space: pre-wrap;
    color: var(--text-color);
}

code {
    background: var(--pre-bg);
    padding: 2px 5px;
    border-radius: 3px;
    font-family: 'Courier New', monospace;
    color: var(--text-color);
}

blockquote {
    border-left: 4px solid var(--blockquote-border);
    padding-left: 15px;
    margin-left: 0;
    color: var(--secondary-color);
    font-style: italic;
}

.tags {
    margin: 20px 0;
    padding: 10px;
    background: var(--entry-bg);
    border-radius: 5px;
}

.tag {
    display: inline-block;
    background: var(--tag-bg);
    color: var(--tag-color);
    padding: 3px 8px;
    margin: 2px;
    border-radius: 3px;
    font-size: 0.9em;
}

.meta {
    color: var(--secondary-color);
    font-size: 0.9em;
    margin-bottom: 20px;
}

.nav {
    margin-top: 30px;
    padding-top: 20px;
    border-top: 1px solid var(--border-color);
}

.nav a {
    color: var(--accent-color);
    text-decoration: none;
    margin-right: 15px;
}

.nav a:hover {
    text-decoration: underline;
    color: var(--accent-hover);
}

.footer {
    margin-top: 40px;
    padding-top: 20px;
    border-top: 2px solid var(--border-color);
    text-align: center;
    color: var(--secondary-color);
    font-size: 0.9em;
}

/* Estilos específicos para el índice */
.entry-list {
    list-style: none;
    padding: 0;
}

.entry-item {
    margin-bottom: 15px;
    padding: 10px;
    border-left: 4px solid var(--accent-color);
    background: var(--entry-bg);
    border-radius: 4px;
    transition: background-color 0.3s ease;
}

.entry-item:hover {
    background: var(--pre-bg);
}

.entry-date {
    color: var(--secondary-color);
    font-size: 0.9em;
    margin-top: 5px;
}

/* Estilos para enlaces */
a {
    color: var(--accent-color);
    text-decoration: none;
}

a:hover {
    text-decoration: underline;
    color: var(--accent-hover);
}

/* Estilos para protocolos especiales */
.protocol-badge {
    display: inline-block;
    padding: 2px 8px;
    margin-right: 10px;
    border-radius: 3px;
    font-size: 0.8em;
    font-weight: bold;
    color: white;
}

.protocol-badge.gemini {
    background: #3498db;
}

.protocol-badge.gopher {
    background: #e67e22;
}

.protocol-badge.spartan {
    background: #9b59b6;
}

.protocol-badge.finger {
    background: #e74c3c;
}

/* Estilos para imágenes */
img {
    max-width: 100%;
    height: auto;
    border: 2px solid var(--border-color);
    border-radius: 5px;
    margin: 10px 0;
}

/* Responsive */
@media (max-width: 600px) {
    body {
        padding: 10px;
    }
    
    .container {
        padding: 15px;
    }
    
    h1 {
        font-size: 1.5em;
    }
    
    article p {
        text-align: left;
    }
}
EOF

    echo "Creado: $archivo_css"
}

# Procesar el archivo línea por línea
fecha=""
nombre_archivo=""
titulo=""
contenido=""
en_contenido=false
errores_encontrados=0

echo "Leyendo archivo: $ARCHIVO_ENTRADAS"

# Leer el archivo preservando espacios y líneas vacías
while IFS= read -r linea; do
    # Detectar fin de entrada
    if [[ "$linea" == "<END>" ]]; then
        if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
            echo "Fin de entrada encontrado, procesando..."
            if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
                ((errores_encontrados++))
            fi
            # Resetear variables
            fecha=""
            nombre_archivo=""
            titulo=""
            contenido=""
            en_contenido=false
        fi
        continue
    fi
    
    # Saltar comentarios (solo líneas que empiezan con ')
    if [[ "$linea" =~ ^\' ]]; then
        continue
    fi
    
    # Si no tenemos fecha, esta línea es la fecha
    if [[ -z "$fecha" ]]; then
        fecha=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos fecha pero no nombre de archivo, esta línea es el nombre
    if [[ -z "$nombre_archivo" ]]; then
        nombre_archivo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos nombre de archivo pero no título, esta línea es el título
    if [[ -z "$titulo" ]]; then
        titulo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        en_contenido=true
        continue
    fi
    
    # Si estamos en el contenido, PRESERVAR la línea exactamente como está
    if [[ "$en_contenido" == true ]]; then
        if [[ -z "$contenido" ]]; then
            contenido="$linea"
        else
            contenido="$contenido"$'\n'"$linea"
        fi
    fi
    
done < "$ARCHIVO_ENTRADAS"

# Procesar última entrada si no terminó con <END>
if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
    echo "Procesando última entrada..."
    if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
        ((errores_encontrados++))
    fi
fi

# Si hay errores, mostrar advertencia
if [ $errores_encontrados -gt 0 ]; then
    echo ""
    echo "⚠️  ADVERTENCIA: Se encontraron $errores_encontrados errores en las fechas"
    echo "   Algunas entradas no fueron procesadas debido a fechas inválidas"
fi

# Crear el archivo CSS
crear_archivo_css

# Crear el archivo índice principal
archivo_index="$DIRECTORIO_SALIDA/index.html"

echo "Creando índice con ${#entradas_index[@]} entradas visibles..."

cat > "$archivo_index" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>$TITULO_INDICE</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>$TITULO_INDICE</h1>
            <p>$BIENVENIDA_INDICE</p>
        </header>

        <nav>
            <a href="tags.html">Explorar por etiquetas</a>
        </nav>

        <main>
            <h2>Entradas del blog</h2>
EOF

# Ordenar entradas por fecha (más reciente primero) y agregar al índice
if [ ${#entradas_index[@]} -gt 0 ]; then
    echo '            <ul class="entry-list">' >> "$archivo_index"
    printf "%s\n" "${entradas_index[@]}" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        archivo_sin_ext="${nombre_archivo%.*}"
        echo "                <li class=\"entry-item\">" >> "$archivo_index"
        echo "                    <a href=\"$archivo_sin_ext.html\">$titulo</a>" >> "$archivo_index"
        echo "                    <div class=\"entry-date\">$fecha</div>" >> "$archivo_index"
        echo "                </li>" >> "$archivo_index"
    done
    echo '            </ul>' >> "$archivo_index"
else
    echo "            <p>No hay entradas disponibles.</p>" >> "$archivo_index"
fi

cat >> "$archivo_index" << EOF
        </main>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF

# Crear índice de etiquetas (solo con entradas publicadas)
if [ ${#etiquetas[@]} -gt 0 ]; then
    crear_indice_etiquetas
else
    # Crear archivo tags.html vacío
    cat > "$DIRECTORIO_SALIDA/tags.html" << EOF
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Índice de Etiquetas - $TITULO_INDICE</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container">
        <header>
            <h1>Índice de Etiquetas</h1>
        </header>

        <main>
            <p>No hay etiquetas disponibles.</p>
        </main>

        <div class="nav">
            <a href="index.html">Volver al índice principal</a>
        </div>

        <div class="footer">
            <p>$TEXTO_FOOTER</p>
        </div>
    </div>
</body>
</html>
EOF
fi

echo ""
echo "=== PROCESAMIENTO COMPLETADO ==="
echo "Archivos creados en: $DIRECTORIO_SALIDA/"
echo "Entradas en el índice: ${#entradas_index[@]}"
echo "Entradas programadas: ${#entradas_programadas[@]}"
echo "Borradores: ${#borradores[@]}"
echo "Etiquetas procesadas: ${#etiquetas[@]}"
echo "Errores de fecha: $errores_encontrados"
echo "Índice generado: $archivo_index"
echo "Índice de etiquetas: $DIRECTORIO_SALIDA/tags.html"
echo "Archivos de etiquetas: $DIRECTORIO_TAGS/"

# Mostrar resumen de borradores
if [ ${#borradores[@]} -gt 0 ]; then
    echo ""
    echo "=== BORRADORES ==="
    for borrador in "${borradores[@]}"; do
        echo "📝 $borrador"
    done
fi

# Mostrar resumen de entradas programadas ORDENADO por fecha y con días restantes
if [ ${#entradas_programadas[@]} -gt 0 ]; then
    echo ""
    echo "=== ENTRADAS PROGRAMADAS ==="
    
    # Obtener fecha actual en timestamp
    fecha_hoy_ts=$(date +%s)
    
    # Ordenar entradas programadas por fecha (más cercana primero)
    printf "%s\n" "${entradas_programadas[@]}" | sort -t'|' -k1,1 | while IFS='|' read -r fecha nombre_archivo titulo; do
        # Convertir fecha programada a timestamp
        fecha_programada_ts=$(date -d "$fecha" +%s 2>/dev/null)
        
        # Calcular días restantes
        if [[ -n "$fecha_programada_ts" ]]; then
            segundos_restantes=$((fecha_programada_ts - fecha_hoy_ts))
            dias_restantes=$((segundos_restantes / 86400))
            
            if [[ $dias_restantes -eq 1 ]]; then
                texto_dias="(Dentro de 1 día)"
            elif [[ $dias_restantes -eq 0 ]]; then
                texto_dias="(Hoy)"
            else
                texto_dias="(Dentro de $dias_restantes días)"
            fi
        else
            texto_dias="(Fecha inválida)"
        fi
        
        echo "⏰ $nombre_archivo - Visible el $fecha $texto_dias"
    done
fi

# Si hubo errores, salir con código de error
if [ $errores_encontrados -gt 0 ]; then
    exit 1
fi
