#!/bin/bash

# =============================================
# GEMLOGTXT.SH - Generador de Gemlogs
# =============================================
# Versión: 2.3
# Descripción: Script para generar blogs en formato Gemini (.gmi)
#              a partir de archivos de texto con entradas.
# Características:
#   - Generación automática de índice y archivos de entradas
#   - Soporte para etiquetas (tags) y categorías
#   - Entradas programadas y borradores
#   - Enlaces externos/minisites con símbolo identificador
#   - Fechas automáticas y validación
#   - Separación por años en el índice con encabezados ###
# =============================================

VERSION="2.3"

ARCHIVO_ENTRADAS="blog.gemlog"
DIRECTORIO_SALIDA="gemlog"
DIRECTORIO_TAGS="$DIRECTORIO_SALIDA/tags"

# =============================================
# CONFIGURACIÓN PERSONALIZABLE DEL GEMLOG
# =============================================

# Textos del índice
TITULO_INDICE="Gemlog de _-~Caleb~-_ 📝"
BIENVENIDA_INDICE="Este es el gemlog del sitio, espero disfrutes de él:"

# Textos de los archivos de entrada
TEXTO_VOLVER_INDICE="Volver al índice"

# Footer personalizado
TEXTO_FOOTER="Este gemlog fue generado automáticamente con gemlog.txt"

# Símbolo para enlaces externos/minisites
SIMBOLO_ENLACE_EXTERNO="- ⤷"

# =============================================
# FIN DE CONFIGURACIÓN
# =============================================

# Verificar que se proporcionó un archivo
if [ -z "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: Debes especificar un archivo de entradas"
    echo "Uso: $0 archivo_entradas.txt"
    exit 1
fi

# Verificar que el archivo existe
if [ ! -f "$ARCHIVO_ENTRADAS" ]; then
    echo "Error: El archivo '$ARCHIVO_ENTRADAS' no existe"
    exit 1
fi

# Crear directorios de salida si no existen
mkdir -p "$DIRECTORIO_SALIDA"
mkdir -p "$DIRECTORIO_TAGS"

# Array para almacenar información del índice
declare -A etiquetas  # Para el índice de tags
declare -a entradas_index
declare -a entradas_programadas
declare -a borradores
declare -a enlaces_externos

# Función para obtener la fecha de hoy en formato AAAA-MM-DD
obtener_fecha_hoy() {
    date +%Y-%m-%d
}

# Función para validar fecha
validar_fecha() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    
    # Verificar formato AAAA-MM-DD
    if [[ ! "$fecha" =~ ^[0-9]{4}-[0-9]{2}-[0-9]{2}$ ]]; then
        echo "❌ ERROR: Formato de fecha inválido '$fecha' en entrada '$titulo'"
        echo "   El formato debe ser AAAA-MM-DD (ej: 2024-01-15)"
        return 1
    fi
    
    # Verificar que la fecha es válida usando date
    if ! date -d "$fecha" >/dev/null 2>&1; then
        echo "❌ ERROR: Fecha inválida '$fecha' en entrada '$titulo'"
        echo "   La fecha no existe en el calendario"
        return 1
    fi
    
    return 0
}

# Función para comparar fechas (0 si fecha1 <= fecha2, 1 si no)
fecha_menor_o_igual() {
    local fecha1="$1"
    local fecha2="$2"
    
    # Convertir a timestamp para comparación numérica
    local ts1=$(date -d "$fecha1" +%s 2>/dev/null)
    local ts2=$(date -d "$fecha2" +%s 2>/dev/null)
    
    if [[ $ts1 -le $ts2 ]]; then
        return 0  # verdadero
    else
        return 1  # falso
    fi
}

# Función para procesar tags
procesar_etiquetas() {
    local tags_linea="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local fecha="$4"
    
    # Extraer tags (palabras que empiezan con #)
    while [[ $tags_linea =~ (#[[:alnum:]]+) ]]; do
        local tag="${BASH_REMATCH[1]}"
        tags_linea="${tags_linea#*${BASH_REMATCH[1]}}"
        
        # Agregar entrada a este tag
        if [[ -z "${etiquetas[$tag]}" ]]; then
            etiquetas[$tag]="$fecha|$nombre_archivo|$titulo"
        else
            etiquetas[$tag]="${etiquetas[$tag]}"$'\n'"$fecha|$nombre_archivo|$titulo"
        fi
    done
}

# Función para crear archivo individual de etiqueta
crear_archivo_etiqueta() {
    local tag="$1"
    local entradas="$2"
    
    # Quitar el # del nombre del archivo
    local nombre_tag=$(echo "$tag" | sed 's/^#//')
    local archivo_tag="$DIRECTORIO_TAGS/${nombre_tag}.gmi"
    
    cat > "$archivo_tag" << EOF
# $tag

EOF

    # Ordenar entradas por fecha (más reciente primero)
    printf "%s\n" "$entradas" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        echo "=> ../$nombre_archivo $fecha - $titulo" >> "$archivo_tag"
    done

    # Footer
    cat >> "$archivo_tag" << EOF

=> ../index.gmi Volver al índice principal
=> ../tags.gmi Ver todas las etiquetas

\`\`\`$TEXTO_FOOTER
---
$TEXTO_FOOTER
\`\`\`
EOF

    echo "Creado: $archivo_tag"
}

# Función para procesar cada entrada
procesar_entrada() {
    local fecha="$1"
    local nombre_archivo="$2"
    local titulo="$3"
    local contenido="$4"
    local tiene_tags=false
    local tags_linea=""
    local es_enlace_externo=false
    
    local fecha_hoy=$(obtener_fecha_hoy)
    
    echo "Procesando entrada: $fecha, $nombre_archivo, '$titulo'"
    
    # Detectar si es un enlace externo (empieza con [ y termina con ])
    if [[ "$nombre_archivo" =~ ^\[.*\]$ ]]; then
        es_enlace_externo=true
        # Extraer la URL del enlace externo (sin los corchetes)
        local url_enlace=$(echo "$nombre_archivo" | sed 's/^\[//;s/\]$//')
        nombre_archivo="$url_enlace"
        echo "   🔗 ENLACE EXTERNO detectado: $url_enlace"
    fi
    
    # Verificar si es un borrador (case-insensitive)
    if [[ "${fecha,,}" == "draft" ]]; then
        borradores+=("$nombre_archivo")
        echo "📝 BORRADOR: '$titulo' - No se publicará en el índice"
        
        # Verificar que el nombre del archivo tenga extensión .gmi (solo si no es enlace externo)
        if [[ "$es_enlace_externo" == false && ! "$nombre_archivo" =~ \.gmi$ ]]; then
            nombre_archivo="${nombre_archivo}.gmi"
        fi
        
        # Buscar línea de tags (primera línea que empieza con #)
        local temp_file=$(mktemp)
        echo "$contenido" > "$temp_file"
        
        while IFS= read -r linea; do
            if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
                tags_linea="$linea"
                tiene_tags=true
                # Eliminar solo la línea exacta de tags usando grep -v
                contenido=$(grep -vFx "$linea" "$temp_file")
                break
            fi
        done < "$temp_file"
        rm -f "$temp_file"
        
        # Ruta completa del archivo
        local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
        
        # Crear el archivo .gmi preservando el formato original
        {
            echo "# $titulo"
            echo ""
            echo "$contenido"
            echo ""
        } > "$archivo_salida"

        # Añadir tags al final del archivo si existen
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            {
                echo "---"
                echo "Etiquetas: $tags_linea"
            } >> "$archivo_salida"
        fi

        # Añadir enlace al índice
        {
            echo ""
            echo "=> index.gmi $TEXTO_VOLVER_INDICE"
            echo "=> tags.gmi Ver todas las etiquetas"
        } >> "$archivo_salida"
        
        echo "Creado: $archivo_salida"
        return 0
    fi
    
    # Validar fecha antes de procesar (solo si no es borrador)
    if ! validar_fecha "$fecha" "$nombre_archivo" "$titulo"; then
        echo "❌ ENTRADA OMITIDA: '$titulo' - Fecha inválida"
        return 1
    fi
    
    # Verificar que el nombre del archivo tenga extensión .gmi (solo si no es enlace externo)
    if [[ "$es_enlace_externo" == false && ! "$nombre_archivo" =~ \.gmi$ ]]; then
        nombre_archivo="${nombre_archivo}.gmi"
    fi
    
    # Buscar línea de tags (primera línea que empieza con #)
    # Usamos un archivo temporal para preservar el formato original
    local temp_file=$(mktemp)
    echo "$contenido" > "$temp_file"
    
    while IFS= read -r linea; do
        if [[ "$linea" =~ ^# && ! "$linea" =~ ^#\ +[^#] ]]; then
            tags_linea="$linea"
            tiene_tags=true
            # Eliminar solo la línea exacta de tags usando grep -v
            contenido=$(grep -vFx "$linea" "$temp_file")
            break
        fi
    done < "$temp_file"
    rm -f "$temp_file"
    
    # Ruta completa del archivo
    local archivo_salida="$DIRECTORIO_SALIDA/$nombre_archivo"
    
    # Crear el archivo .gmi preservando el formato original
    {
        echo "# $titulo"
        echo ""
        echo "$contenido"
        echo ""
    } > "$archivo_salida"

    # Añadir tags al final del archivo si existen
    if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
        {
            echo "---"
            echo "Etiquetas: $tags_linea"
        } >> "$archivo_salida"
    fi

    # Añadir enlace al índice
    {
        echo ""
        echo "=> index.gmi $TEXTO_VOLVER_INDICE"
        echo "=> tags.gmi Ver todas las etiquetas"
    } >> "$archivo_salida"
    
    echo "Creado: $archivo_salida"
    
    # Solo agregar al índice y procesar tags si la fecha es hoy o pasada
    if fecha_menor_o_igual "$fecha" "$fecha_hoy"; then
        # Para enlaces externos, añadir símbolo al título
        if [[ "$es_enlace_externo" == true ]]; then
            titulo_con_formato="$titulo $SIMBOLO_ENLACE_EXTERNO"
            enlaces_externos+=("$nombre_archivo")
        else
            titulo_con_formato="$titulo"
        fi
        
        entradas_index+=("$fecha|$nombre_archivo|$titulo_con_formato")
        
        # Procesar tags solo para entradas publicadas
        if [[ "$tiene_tags" == true && -n "$tags_linea" ]]; then
            echo "   Tags encontrados: $tags_linea"
            procesar_etiquetas "$tags_linea" "$nombre_archivo" "$titulo" "$fecha"
        fi
        echo "✅ Añadida al índice: $titulo"
    else
        entradas_programadas+=("$fecha|$nombre_archivo|$titulo")
        echo "⏰ Programada para el futuro: $titulo (visible el $fecha)"
        # NOTA: Los tags de entradas programadas NO se procesan para el índice de tags
    fi
    
    return 0
}

# Función para ordenar etiquetas por uso (de más a menos)
obtener_etiquetas_ordenadas() {
    for tag in "${!etiquetas[@]}"; do
        local count=$(echo "${etiquetas[$tag]}" | wc -l)
        echo "$count|$tag"
    done | sort -nr | cut -d'|' -f2
}

# Función para crear el índice de etiquetas
crear_indice_etiquetas() {
    local archivo_tags="$DIRECTORIO_SALIDA/tags.gmi"
    
    cat > "$archivo_tags" << EOF
# Índice de Etiquetas

Explora las entradas por etiquetas:

EOF

    # Ordenar tags por uso (de más a menos)
    if [ ${#etiquetas[@]} -gt 0 ]; then
        while IFS= read -r tag; do
            # Contar número de entradas para este tag
            local count=$(echo "${etiquetas[$tag]}" | wc -l)
            local nombre_tag=$(echo "$tag" | sed 's/^#//')
            echo "=> tags/${nombre_tag}.gmi $tag ($count)" >> "$archivo_tags"
            
            # Crear archivo individual para esta etiqueta
            crear_archivo_etiqueta "$tag" "${etiquetas[$tag]}"
        done < <(obtener_etiquetas_ordenadas)
    else
        echo "No hay etiquetas disponibles." >> "$archivo_tags"
    fi

    # Footer
    cat >> "$archivo_tags" << EOF

=> index.gmi Volver al índice principal

\`\`\`$TEXTO_FOOTER
---
$TEXTO_FOOTER
\`\`\`
EOF

    echo "Creado: $archivo_tags"
}

# Procesar el archivo línea por línea - VERSIÓN CORREGIDA
fecha=""
nombre_archivo=""
titulo=""
contenido=""
en_contenido=false
errores_encontrados=0

echo "Leyendo archivo: $ARCHIVO_ENTRADAS"

# Leer el archivo preservando espacios y líneas vacías
while IFS= read -r linea; do
    # PRESERVAR la línea original sin modificar espacios
    
    # Detectar fin de entrada
    if [[ "$linea" == "<END>" ]]; then
        if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
            echo "Fin de entrada encontrado, procesando..."
            if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
                ((errores_encontrados++))
            fi
            # Resetear variables
            fecha=""
            nombre_archivo=""
            titulo=""
            contenido=""
            en_contenido=false
        fi
        continue
    fi
    
    # Saltar comentarios (solo líneas que empiezan con ')
    if [[ "$linea" =~ ^\' ]]; then
        continue
    fi
    
    # Si no tenemos fecha, esta línea es la fecha (limpiar solo para metadatos)
    if [[ -z "$fecha" ]]; then
        fecha=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos fecha pero no nombre de archivo, esta línea es el nombre
    if [[ -z "$nombre_archivo" ]]; then
        nombre_archivo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        continue
    fi
    
    # Si tenemos nombre de archivo pero no título, esta línea es el título
    if [[ -z "$titulo" ]]; then
        titulo=$(echo "$linea" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        en_contenido=true
        continue
    fi
    
    # Si estamos en el contenido, PRESERVAR la línea exactamente como está
    if [[ "$en_contenido" == true ]]; then
        if [[ -z "$contenido" ]]; then
            contenido="$linea"
        else
            contenido="$contenido"$'\n'"$linea"
        fi
    fi
    
done < "$ARCHIVO_ENTRADAS"

# Procesar última entrada si no terminó con <END>
if [[ -n "$fecha" && -n "$nombre_archivo" && -n "$titulo" ]]; then
    echo "Procesando última entrada..."
    if ! procesar_entrada "$fecha" "$nombre_archivo" "$titulo" "$contenido"; then
        ((errores_encontrados++))
    fi
fi

# Si hay errores, mostrar advertencia
if [ $errores_encontrados -gt 0 ]; then
    echo ""
    echo "⚠️  ADVERTENCIA: Se encontraron $errores_encontrados errores en las fechas"
    echo "   Algunas entradas no fueron procesadas debido a fechas inválidas"
fi

# Crear el archivo índice principal
archivo_index="$DIRECTORIO_SALIDA/index.gmi"

echo "Creando índice con ${#entradas_index[@]} entradas visibles..."

cat > "$archivo_index" << EOF
# $TITULO_INDICE

$BIENVENIDA_INDICE

=> tags.gmi Explorar por etiquetas

EOF

# Mostrar todas las entradas en orden cronológico inverso (más reciente primero)
# con separación por años usando ###
if [ ${#entradas_index[@]} -gt 0 ]; then
    anio_actual=""
    printf "%s\n" "${entradas_index[@]}" | sort -r | while IFS='|' read -r fecha nombre_archivo titulo; do
        anio_entrada=$(echo "$fecha" | cut -d'-' -f1)
        
        # Si cambia el año, añadir encabezado ###
        if [[ "$anio_entrada" != "$anio_actual" ]]; then
            echo "### $anio_entrada" >> "$archivo_index"
            anio_actual="$anio_entrada"
        fi
        
        echo "=> $nombre_archivo $fecha - $titulo" >> "$archivo_index"
    done
else
    echo "No hay entradas disponibles." >> "$archivo_index"
fi

# Footer
cat >> "$archivo_index" << EOF

\`\`\`$TEXTO_FOOTER
---
$TEXTO_FOOTER
\`\`\`
EOF

# Crear índice de etiquetas (solo con entradas publicadas)
if [ ${#etiquetas[@]} -gt 0 ]; then
    crear_indice_etiquetas
else
    # Crear archivo tags.gmi vacío
    cat > "$DIRECTORIO_SALIDA/tags.gmi" << EOF
# Índice de Etiquetas

No hay etiquetas disponibles.

=> index.gmi Volver al índice principal

\`\`\`$TEXTO_FOOTER
---
$TEXTO_FOOTER
\`\`\`
EOF
fi

echo ""
echo "=== PROCESAMIENTO COMPLETADO ==="
echo "Archivos creados en: $DIRECTORIO_SALIDA/"
echo "Versión del script: $VERSION"
echo "Entradas en el índice: ${#entradas_index[@]}"
echo "Enlaces externos: ${#enlaces_externos[@]}"
echo "Entradas programadas: ${#entradas_programadas[@]}"
echo "Borradores: ${#borradores[@]}"
echo "Etiquetas procesadas: ${#etiquetas[@]}"
echo "Errores de fecha: $errores_encontrados"
echo "Índice generado: $archivo_index"
echo "Índice de etiquetas: $DIRECTORIO_SALIDA/tags.gmi"
echo "Archivos de etiquetas: $DIRECTORIO_TAGS/"

# Mostrar resumen de borradores
if [ ${#borradores[@]} -gt 0 ]; then
    echo ""
    echo "=== BORRADORES ==="
    for borrador in "${borradores[@]}"; do
        echo "📝 $borrador"
    done
fi

# Mostrar resumen de enlaces externos
if [ ${#enlaces_externos[@]} -gt 0 ]; then
    echo ""
    echo "=== ENLACES EXTERNOS ==="
    for enlace in "${enlaces_externos[@]}"; do
        echo "🔗 $enlace"
    done
fi

# Mostrar resumen de entradas programadas ORDENADO por fecha y con días restantes
if [ ${#entradas_programadas[@]} -gt 0 ]; then
    echo ""
    echo "=== ENTRADAS PROGRAMADAS ==="
    
    # Obtener fecha actual en timestamp
    fecha_hoy_ts=$(date +%s)
    
    # Ordenar entradas programadas por fecha (más cercana primero)
    printf "%s\n" "${entradas_programadas[@]}" | sort -t'|' -k1,1 | while IFS='|' read -r fecha nombre_archivo titulo; do
        # Convertir fecha programada a timestamp
        fecha_programada_ts=$(date -d "$fecha" +%s 2>/dev/null)
        
        # Calcular días restantes
        if [[ -n "$fecha_programada_ts" ]]; then
            segundos_restantes=$((fecha_programada_ts - fecha_hoy_ts))
            dias_restantes=$((segundos_restantes / 86400))
            
            if [[ $dias_restantes -eq 1 ]]; then
                texto_dias="(Dentro de 1 día)"
            elif [[ $dias_restantes -eq 0 ]]; then
                texto_dias="(Hoy)"
            else
                texto_dias="(Dentro de $dias_restantes días)"
            fi
        else
            texto_dias="(Fecha inválida)"
        fi
        
        echo "⏰ $nombre_archivo - Visible el $fecha $texto_dias"
    done
fi

# Si hubo errores, salir con código de error
if [ $errores_encontrados -gt 0 ]; then
    exit 1
fi

# =============================================
# LICENCIA
# =============================================
# This is free and unencumbered software released into the public domain.
#
# Anyone is free to copy, modify, publish, use, compile, sell, or
# distribute this software, either in source code form or as a compiled
# binary, for any purpose, commercial or non-commercial, and by any
# means.
#
# In jurisdictions that recognize copyright laws, the author or authors
# of this software dedicate any and all copyright interest in the
# software to the public domain. We make this dedication for the benefit
# of the public at large and to the detriment of our heirs and
# successors. We intend this dedication to be an overt act of
# relinquishment in perpetuity of all present and future rights to this
# software under copyright law.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.
#
# For more information, please refer to <http://unlicense.org/>
# =============================================
