#!/bin/bash

# generar_atom.sh
# Uso: generar_atom.sh directorio_html [base_url]

DIRECTORIO_HTML="$1"
BASE_URL="${2:-https://misitio.com}"

# =============================================
# CONFIGURACIÓN DEL FEED ATOM
# =============================================

TITULO_FEED="Blog de _-~Caleb~-_ 📝"
AUTOR_NOMBRE="Caleb"
AUTOR_EMAIL="caleb@misitio.com"  # Opcional
ID_FEED="$BASE_URL/"
DESCRIPCION_FEED="Blog personal con entradas sobre tecnología, pensamientos y más"

# =============================================
# FIN DE CONFIGURACIÓN
# =============================================

if [ -z "$DIRECTORIO_HTML" ]; then
    echo "Error: Debes especificar un directorio HTML"
    echo "Uso: $0 directorio_html [base_url]"
    exit 1
fi

if [ ! -d "$DIRECTORIO_HTML" ]; then
    echo "Error: El directorio '$DIRECTORIO_HTML' no existe"
    exit 1
fi

ARCHIVO_INDEX="$DIRECTORIO_HTML/index.html"
ARCHIVO_ATOM="$DIRECTORIO_HTML/atom.xml"

if [ ! -f "$ARCHIVO_INDEX" ]; then
    echo "Error: No se encontró index.html en $DIRECTORIO_HTML"
    exit 1
fi

# Función para detectar si es HTML3
es_html3() {
    local archivo="$1"
    if grep -q "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">" "$archivo" 2>/dev/null; then
        return 0  # Es HTML3
    else
        return 1  # Es HTML5
    fi
}

# Función para extraer entradas del index.html
extraer_entradas_del_index() {
    local archivo_index="$1"
    local archivo_temp="$2"
    
    if es_html3 "$archivo_index"; then
        echo "Procesando índice HTML3..."
        # Para HTML3: buscar patrones específicos
        grep -A 5 'entry-item' "$archivo_index" | while IFS= read -r linea; do
            # Buscar enlace
            if echo "$linea" | grep -q 'HREF='; then
                archivo_enlace=$(echo "$linea" | sed -n 's/.*HREF="\([^"]*\)".*/\1/p')
                titulo=$(echo "$linea" | sed -n 's/.*<STRONG>\([^<]*\)<\/STRONG>.*/\1/p')
                if [ -n "$archivo_enlace" ] && [ -n "$titulo" ]; then
                    echo "Encontrado enlace: $archivo_enlace - $titulo"
                fi
            fi
            
            # Buscar fecha
            if echo "$linea" | grep -q 'entry-date'; then
                fecha=$(echo "$linea" | sed -n 's/.*>\([0-9]\{4\}-[0-9]\{2\}-[0-9]\{2\}\)<.*/\1/p')
                if [ -n "$fecha" ] && [ -n "$archivo_enlace" ] && [ -n "$titulo" ]; then
                    echo "$fecha|$archivo_enlace|$titulo" >> "$archivo_temp"
                    echo "✓ Entrada añadida: $fecha - $titulo"
                    # Resetear variables
                    fecha=""
                    archivo_enlace=""
                    titulo=""
                fi
            fi
        done
    else
        echo "Procesando índice HTML5..."
        # Para HTML5: buscar patrones específicos
        grep -A 5 'entry-item' "$archivo_index" | while IFS= read -r linea; do
            # Buscar enlace
            if echo "$linea" | grep -q 'href='; then
                archivo_enlace=$(echo "$linea" | sed -n 's/.*href="\([^"]*\)".*/\1/p')
                titulo=$(echo "$linea" | sed -n 's/.*>\([^<]*\)<.*/\1/p')
                if [ -n "$archivo_enlace" ] && [ -n "$titulo" ]; then
                    echo "Encontrado enlace: $archivo_enlace - $titulo"
                fi
            fi
            
            # Buscar fecha
            if echo "$linea" | grep -q 'entry-date'; then
                fecha=$(echo "$linea" | sed -n 's/.*>\([0-9]\{4\}-[0-9]\{2\}-[0-9]\{2\}\)<.*/\1/p')
                if [ -n "$fecha" ] && [ -n "$archivo_enlace" ] && [ -n "$titulo" ]; then
                    echo "$fecha|$archivo_enlace|$titulo" >> "$archivo_temp"
                    echo "✓ Entrada añadida: $fecha - $titulo"
                    # Resetear variables
                    fecha=""
                    archivo_enlace=""
                    titulo=""
                fi
            fi
        done
    fi
}

# Función para generar resumen del contenido
generar_resumen() {
    local archivo="$1"
    local contenido=""
    
    if [ ! -f "$archivo" ]; then
        echo "Entrada del blog"
        return
    fi
    
    # Extraer primer párrafo que no sea vacío
    contenido=$(grep '<p>' "$archivo" | grep -v '^<p></p>$' | head -1 | sed 's/<p>//;s/<\/p>//')
    
    # Si no encontramos, buscar cualquier texto
    if [ -z "$contenido" ] || [ "${#contenido}" -lt 10 ]; then
        contenido=$(grep -v -E '^<[a-z]|^</[a-z]|^<!DOCTYPE|^<HTML|^<HEAD|^<TITLE|^<BODY|^<CENTER|^<TABLE|^<TR|^<TD' "$archivo" | head -3 | tr '\n' ' ' | cut -c1-150)
    fi
    
    # Limpiar y escapar caracteres especiales para XML
    if [ -n "$contenido" ]; then
        contenido=$(echo "$contenido" | sed 's/&/&amp;/g; s/</&lt;/g; s/>/&gt;/g; s/"/&quot;/g; s/'"'"'/&apos;/g')
    else
        contenido="Entrada del blog"
    fi
    
    echo "$contenido"
}

# Crear archivo temporal para almacenar entradas
ARCHIVO_TEMP=$(mktemp)

echo "Extrayendo entradas del índice: $ARCHIVO_INDEX"

# Extraer entradas del index.html
extraer_entradas_del_index "$ARCHIVO_INDEX" "$ARCHIVO_TEMP"

# Verificar si encontramos entradas
if [ ! -s "$ARCHIVO_TEMP" ]; then
    echo "ERROR: No se encontraron entradas en el índice"
    echo "Intentando método alternativo..."
    
    # Método alternativo: buscar directamente en el índice
    if es_html3 "$ARCHIVO_INDEX"; then
        # Para HTML3
        grep 'HREF=' "$ARCHIVO_INDEX" | while IFS= read -r linea; do
            archivo_enlace=$(echo "$linea" | sed -n 's/.*HREF="\([^"]*\)".*/\1/p')
            titulo=$(echo "$linea" | sed -n 's/.*<STRONG>\([^<]*\)<\/STRONG>.*/\1/p')
            if [[ "$archivo_enlace" =~ \.html$ ]] && [ -n "$titulo" ]; then
                # Buscar fecha cerca de este enlace
                fecha=$(grep -A 10 "$titulo" "$ARCHIVO_INDEX" | grep -o '[0-9]\{4\}-[0-9]\{2\}-[0-9]\{2\}' | head -1)
                if [ -n "$fecha" ]; then
                    echo "$fecha|$archivo_enlace|$titulo" >> "$ARCHIVO_TEMP"
                    echo "✓ Entrada añadida (método alternativo): $fecha - $titulo"
                fi
            fi
        done
    else
        # Para HTML5
        grep 'href=' "$ARCHIVO_INDEX" | grep '\.html' | while IFS= read -r linea; do
            archivo_enlace=$(echo "$linea" | sed -n 's/.*href="\([^"]*\)".*/\1/p')
            titulo=$(echo "$linea" | sed -n 's/.*>\([^<]*\)<.*/\1/p')
            if [ -n "$archivo_enlace" ] && [ -n "$titulo" ] && [ "$titulo" != "Explorar por etiquetas" ]; then
                # Buscar fecha cerca de este enlace
                fecha=$(grep -A 10 "$titulo" "$ARCHIVO_INDEX" | grep -o '[0-9]\{4\}-[0-9]\{2\}-[0-9]\{2\}' | head -1)
                if [ -n "$fecha" ]; then
                    echo "$fecha|$archivo_enlace|$titulo" >> "$ARCHIVO_TEMP"
                    echo "✓ Entrada añadida (método alternativo): $fecha - $titulo"
                fi
            fi
        done
    fi
fi

# Verificar nuevamente si encontramos entradas
if [ ! -s "$ARCHIVO_TEMP" ]; then
    echo "ERROR: No se pudieron extraer entradas del índice"
    echo "El formato del index.html podría ser diferente"
    rm -f "$ARCHIVO_TEMP"
    exit 1
fi

# Ordenar entradas por fecha (más reciente primero) y limitar a 15
ENTRADAS_ORDENADAS=$(sort -r "$ARCHIVO_TEMP" | head -15)
TOTAL_ENTRADAS=$(echo "$ENTRADAS_ORDENADAS" | wc -l)

echo "Encontradas $TOTAL_ENTRADAS entradas en el índice"

# Obtener fecha de actualización del feed (fecha de la entrada más reciente)
FECHA_ACTUALIZACION=$(echo "$ENTRADAS_ORDENADAS" | head -1 | cut -d'|' -f1)
if [ -n "$FECHA_ACTUALIZACION" ]; then
    FECHA_ACTUALIZACION="${FECHA_ACTUALIZACION}T12:00:00Z"
else
    FECHA_ACTUALIZACION=$(date -u +%Y-%m-%dT%H:%M:%SZ)
fi

FECHA_GENERACION=$(date -u +%Y-%m-%dT%H:%M:%SZ)

echo "Generando feed Atom con $TOTAL_ENTRADAS entradas..."

cat > "$ARCHIVO_ATOM" << EOF
<?xml version="1.0" encoding="utf-8"?>
<feed xmlns="http://www.w3.org/2005/Atom">
  <title>$TITULO_FEED</title>
  <link href="$BASE_URL/atom.xml" rel="self"/>
  <link href="$BASE_URL/"/>
  <updated>$FECHA_ACTUALIZACION</updated>
  <id>$ID_FEED</id>
  <author>
    <name>$AUTOR_NOMBRE</name>
    $( [ -n "$AUTOR_EMAIL" ] && echo "<email>$AUTOR_EMAIL</email>" )
  </author>
  <subtitle>$DESCRIPCION_FEED</subtitle>
  <generator>gemlogtxt2html + generar_atom.sh</generator>
EOF

# Procesar entradas ordenadas
CONTADOR=0
while IFS='|' read -r fecha archivo_enlace titulo; do
    if [ -n "$fecha" ] && [ -n "$archivo_enlace" ] && [ -n "$titulo" ]; then
        # Construir ruta completa al archivo
        archivo_completo="$DIRECTORIO_HTML/$archivo_enlace"
        fecha_atom="${fecha}T12:00:00Z"
        resumen=$(generar_resumen "$archivo_completo")
        
        # Escapar caracteres especiales en el título
        titulo_escape=$(echo "$titulo" | sed 's/&/&amp;/g; s/</&lt;/g; s/>/&gt;/g; s/"/&quot;/g; s/'"'"'/&apos;/g')
        
        cat >> "$ARCHIVO_ATOM" << ENTRY
  <entry>
    <title>$titulo_escape</title>
    <link href="$BASE_URL/$archivo_enlace"/>
    <id>$BASE_URL/${archivo_enlace%.*}</id>
    <updated>$fecha_atom</updated>
    <published>$fecha_atom</published>
    <author>
      <name>$AUTOR_NOMBRE</name>
    </author>
    <summary type="html"><![CDATA[${resumen}...]]></summary>
  </entry>
ENTRY
        
        CONTADOR=$((CONTADOR + 1))
        echo "✓ Añadida al feed: $titulo ($fecha)"
    fi
done <<< "$ENTRADAS_ORDENADAS"

# Cerrar feed
cat >> "$ARCHIVO_ATOM" << EOF
</feed>
EOF

# Limpiar archivo temporal
rm -f "$ARCHIVO_TEMP"

echo ""
echo "=== FEED ATOM GENERADO ==="
echo "Archivo: $ARCHIVO_ATOM"
echo "Entradas incluidas: $CONTADOR"
echo "Base URL: $BASE_URL"
echo "Fecha de actualización: $FECHA_ACTUALIZACION"

# Mostrar contenido del feed para verificación
echo ""
echo "=== PRIMERAS LÍNEAS DEL FEED ==="
head -20 "$ARCHIVO_ATOM"
