#!/bin/bash

# SISTEMA COMPLETO DE ETIQUETAS PARA CHIMAQUE - VERSIÓN FINAL CORREGIDA
ARCHIVO_DATOS="capsulas.txt"
DIR_ETIQUETAS="etiquetas"
ETIQUETAS_POR_PAGINA=100

# FUNCIÓN: Generar índice de etiquetas (etiquetas.gmi) - CON FORMATO EXACTO
generar_indice_etiquetas() {
    echo "📋 Generando índice de etiquetas..."
    
    # Archivo temporal para procesar
    ARCHIVO_TEMP=$(mktemp)

    # Procesar capsulas.txt para extraer etiquetas únicas (ignorando comentarios)
    grep -v '^[[:space:]]*#' "$ARCHIVO_DATOS" | grep -v '^[[:space:]]*$' | awk -F'|' '
    {
        if (NF >= 4) {
            split($4, tags, ",")
            for (i in tags) {
                # Limpiar espacios
                gsub(/^[ \t]+|[ \t]+$/, "", tags[i])
                if (tags[i] != "" && length(tags[i]) > 0) {
                    print tolower(tags[i])
                }
            }
        }
    }' | sort | uniq -c | sort -nr > "$ARCHIVO_TEMP"

    TOTAL_ETIQUETAS=$(wc -l < "$ARCHIVO_TEMP" 2>/dev/null || echo "0")
    TOTAL_PAGINAS=$(( (TOTAL_ETIQUETAS + ETIQUETAS_POR_PAGINA - 1) / ETIQUETAS_POR_PAGINA ))

    # Generar páginas de etiquetas
    for ((PAGINA=1; PAGINA<=TOTAL_PAGINAS; PAGINA++)); do
        if [ $PAGINA -eq 1 ]; then
            ARCHIVO_SALIDA="etiquetas.gmi"
        else
            ARCHIVO_SALIDA="etiquetas-${PAGINA}.gmi"
        fi
        
        # Calcular rangos para la página actual
        INICIO=$(( (PAGINA - 1) * ETIQUETAS_POR_PAGINA + 1 ))
        FIN=$(( PAGINA * ETIQUETAS_POR_PAGINA ))
        if [ $FIN -gt $TOTAL_ETIQUETAS ]; then
            FIN=$TOTAL_ETIQUETAS
        fi
        ETIQUETAS_EN_PAGINA=$(( FIN - INICIO + 1 ))
        
        # Crear el archivo de la página CON FORMATO EXACTO
        echo "# 🏷️ Etiquetas del Directorio" > "$ARCHIVO_SALIDA"
        echo "" >> "$ARCHIVO_SALIDA"
        echo "=> index.gmi ← Volver al directorio principal" >> "$ARCHIVO_SALIDA"
        echo "" >> "$ARCHIVO_SALIDA"
        echo "Total de etiquetas únicas: $TOTAL_ETIQUETAS" >> "$ARCHIVO_SALIDA"
        echo "Etiquetas mostradas: $ETIQUETAS_EN_PAGINA" >> "$ARCHIVO_SALIDA"
        echo "" >> "$ARCHIVO_SALIDA"
        
        # LISTA DE ETIQUETAS
        echo "Lista de etiquetas utilizadas en las cápsulas indexadas:" >> "$ARCHIVO_SALIDA"
        echo "" >> "$ARCHIVO_SALIDA"

        # Procesar etiquetas para esta página
        contador_etiquetas=0
        etiquetas_mostradas=0
        
        while read count etiqueta && [ $contador_etiquetas -lt $FIN ]; do
            contador_etiquetas=$((contador_etiquetas + 1))
            
            # Saltar etiquetas de páginas anteriores
            if [ $contador_etiquetas -lt $INICIO ]; then
                continue
            fi
            
            etiqueta_limpia=$(echo "$etiqueta" | xargs | sed 's/[^a-zA-Z0-9áéíóúñü_]//g')
            
            # Verificar que es una sola palabra válida
            if [[ -n "$etiqueta_limpia" ]] && [[ "$etiqueta_limpia" =~ ^[^[:space:]]+$ ]]; then
                echo "=> etiquetas/${etiqueta_limpia}.gmi #${etiqueta_limpia} ($count)" >> "$ARCHIVO_SALIDA"
                etiquetas_mostradas=$((etiquetas_mostradas + 1))
            fi
        done < "$ARCHIVO_TEMP"
        
        echo "" >> "$ARCHIVO_SALIDA"
        echo "---" >> "$ARCHIVO_SALIDA"
        echo "=> index.gmi ← Volver al directorio principal" >> "$ARCHIVO_SALIDA"
        
        echo "✅ $ARCHIVO_SALIDA generado con $etiquetas_mostradas etiquetas"
    done

    # Limpiar archivo temporal
    rm -f "$ARCHIVO_TEMP"
    
    echo $TOTAL_ETIQUETAS
}

# FUNCIÓN: Generar páginas individuales de etiquetas (CON TOTAL ARRIBA)
generar_paginas_individuales() {
    echo "🔍 Generando páginas individuales de etiquetas..."
    
    # Archivo temporal filtrado (sin comentarios ni líneas vacías)
    ARCHIVO_FILTRADO=$(mktemp)
    grep -v '^[[:space:]]*#' "$ARCHIVO_DATOS" | grep -v '^[[:space:]]*$' > "$ARCHIVO_FILTRADO"

    # Procesar cada línea del archivo de datos filtrado
    while IFS='|' read -r url titulo idioma tags; do
        # Limpiar datos
        url=$(echo "$url" | xargs)
        titulo=$(echo "$titulo" | xargs)
        idioma=$(echo "$idioma" | xargs | tr '[:upper:]' '[:lower:]')
        tags=$(echo "$tags" | xargs | tr '[:upper:]' '[:lower:]')
        
        # Saltar líneas que no tengan los campos necesarios
        if [[ -z "$url" ]] || [[ -z "$titulo" ]] || [[ -z "$tags" ]]; then
            continue
        fi
        
        # Procesar cada etiqueta individualmente
        IFS=',' read -ra array_etiquetas <<< "$tags"
        
        for etiqueta in "${array_etiquetas[@]}"; do
            etiqueta_limpia=$(echo "$etiqueta" | xargs | sed 's/[^a-zA-Z0-9áéíóúñü_]//g')
            
            # Verificar que es una sola palabra válida
            if [[ -n "$etiqueta_limpia" ]] && [[ "$etiqueta_limpia" =~ ^[^[:space:]]+$ ]]; then
                archivo_salida="${DIR_ETIQUETAS}/${etiqueta_limpia}.gmi"
                
                # Si el archivo no existe, crear encabezado
                if [ ! -f "$archivo_salida" ]; then
                    echo "# 🏷️ Cápsulas con #${etiqueta_limpia}" > "$archivo_salida"
                    echo "" >> "$archivo_salida"
                    echo "## Cápsulas etiquetadas como #${etiqueta_limpia}:" >> "$archivo_salida"
                    echo "" >> "$archivo_salida"
                fi
                
                # Añadir entrada al archivo de etiqueta (evitar duplicados)
                if ! grep -q "=> $url " "$archivo_salida"; then
                    # Limitar a máximo 5 etiquetas para esta cápsula
                    otras_etiquetas=""
                    contador=0
                    for otra_etiqueta in "${array_etiquetas[@]}"; do
                        otra_etiqueta_limpia=$(echo "$otra_etiqueta" | xargs | sed 's/[^a-zA-Z0-9áéíóúñü_]//g')
                        if [[ "$otra_etiqueta_limpia" != "$etiqueta_limpia" ]] && \
                           [[ -n "$otra_etiqueta_limpia" ]] && [[ "$otra_etiqueta_limpia" =~ ^[^[:space:]]+$ ]] && \
                           [[ $contador -lt 5 ]]; then
                            if [ -z "$otras_etiquetas" ]; then
                                otras_etiquetas="#$otra_etiqueta_limpia"
                            else
                                otras_etiquetas="$otras_etiquetas #$otra_etiqueta_limpia"
                            fi
                            contador=$((contador + 1))
                        fi
                    done
                    
                    # Escribir enlace SIN línea vacía después
                    echo "=> $url $titulo [$idioma] $otras_etiquetas" >> "$archivo_salida"
                fi
            fi
        done
        
    done < "$ARCHIVO_FILTRADO"

    # Añadir footer a cada archivo de etiqueta CON EL TOTAL ARRIBA
    for archivo_etiqueta in "$DIR_ETIQUETAS"/*.gmi; do
        if [ -f "$archivo_etiqueta" ]; then
            # Contar enlaces correctamente
            total_enlaces=$(grep -E -c "=> [a-zA-Z][a-zA-Z0-9+.-]*://" "$archivo_etiqueta")
            
            # Crear archivo temporal con nuevo formato
            ARCHIVO_TEMP=$(mktemp)
            
            # Copiar encabezado
            head -n 4 "$archivo_etiqueta" > "$ARCHIVO_TEMP"
            
            # Añadir total en formato texto preformateado
            echo "\`\`\`" >> "$ARCHIVO_TEMP"
            echo "**Total: $total_enlaces cápsulas**" >> "$ARCHIVO_TEMP"
            echo "\`\`\`" >> "$ARCHIVO_TEMP"
            echo "" >> "$ARCHIVO_TEMP"
            
            # Copiar el contenido (enlaces) saltando las primeras 4 líneas (encabezado)
            tail -n +5 "$archivo_etiqueta" >> "$ARCHIVO_TEMP"
            
            # Reemplazar el archivo original
            mv "$ARCHIVO_TEMP" "$archivo_etiqueta"
            
            # Añadir footer al final
            echo "" >> "$archivo_etiqueta"
            echo "---" >> "$archivo_etiqueta"
            echo "=> ../etiquetas.gmi ← Ver todas las etiquetas" >> "$archivo_etiqueta"
            echo "=> ../index.gmi ← Volver al directorio principal" >> "$archivo_etiqueta"
            
            echo "✅ $(basename "$archivo_etiqueta"): $total_enlaces cápsulas"
        fi
    done

    # Limpiar archivo temporal
    rm -f "$ARCHIVO_FILTRADO"
}

# PROGRAMA PRINCIPAL
echo "🚀 INICIANDO SISTEMA DE ETIQUETAS"
echo "================================="

# Verificar que el archivo existe
if [ ! -f "$ARCHIVO_DATOS" ]; then
    echo "❌ Error: No se encuentra el archivo $ARCHIVO_DATOS"
    exit 1
fi

# Crear directorio si no existe
mkdir -p "$DIR_ETIQUETAS"

# Limpiar etiquetas anteriores
echo "🧹 Limpiando etiquetas anteriores..."
rm -f "$DIR_ETIQUETAS"/*.gmi 2>/dev/null
rm -f etiquetas*.gmi 2>/dev/null

# Generar todo el sistema
TOTAL_ETIQUETAS=$(generar_indice_etiquetas)
generar_paginas_individuales

echo ""
echo "🎉 SISTEMA DE ETIQUETAS COMPLETADO"
echo "================================="
echo "   - Total etiquetas únicas: $TOTAL_ETIQUETAS"
echo "   - Páginas de índice generadas en: etiquetas.gmi, etiquetas-*.gmi"
echo "   - Páginas individuales generadas en: $DIR_ETIQUETAS/"
