#!/bin/bash

# Versión funcional con paginación y validación - Directorio Gemini
ARCHIVO_DATOS="capsulas.txt"
VERSION="1.0e"  # ← AÑADIDA VARIABLE DE VERSIÓN

# Verificar que el archivo existe
if [ ! -f "$ARCHIVO_DATOS" ]; then
    echo "❌ Error: No se encuentra el archivo $ARCHIVO_DATOS"
    exit 1
fi

# Obtener fecha actual en formato DD-MM-AAAA
FECHA_ACTUAL=$(date '+%d-%m-%Y')  # ← AÑADIDA FECHA ACTUAL

# Filtrar líneas: eliminar vacías, comentarios y procesar
ARCHIVO_TEMP=$(mktemp)
ARCHIVO_UNICO=$(mktemp)

echo "🔍 Procesando archivo y eliminando duplicados..."
grep -v '^[[:space:]]*$' "$ARCHIVO_DATOS" | grep -v '^[[:space:]]*#' | while IFS='|' read -r url titulo idioma tags; do
    # Limpiar URL para comparación
    url_limpia=$(echo "$url" | xargs)
    
    # Verificar si la URL ya fue procesada
    if ! grep -q "^${url_limpia}$" "$ARCHIVO_UNICO"; then
        # Si es única, guardar en archivo único y procesar
        echo "$url_limpia" >> "$ARCHIVO_UNICO"
        echo "$url|$titulo|$idioma|$tags" >> "$ARCHIVO_TEMP"
    else
        echo "⚠️  ADVERTENCIA: URL duplicada ignorada - $url_limpia"
    fi
done

TOTAL_UNICOS=$(wc -l < "$ARCHIVO_TEMP" 2>/dev/null || echo "0")
TOTAL_ORIGINAL=$(grep -v '^[[:space:]]*$' "$ARCHIVO_DATOS" | grep -v '^[[:space:]]*#' | wc -l)
DUPLICADOS=$((TOTAL_ORIGINAL - TOTAL_UNICOS))

if [ $DUPLICADOS -gt 0 ]; then
    echo "⚠️  Se ignoraron $DUPLICADOS enlace(s) duplicado(s)"
    echo ""
fi

# Configuración de paginación
ENLACES_POR_PAGINA=100
TOTAL_PAGINAS=$(( (TOTAL_UNICOS + ENLACES_POR_PAGINA - 1) / ENLACES_POR_PAGINA ))

# Crear archivo temporal para ordenar por primera etiqueta
TEMP_FILE=$(mktemp)

# Procesar y ordenar por primera etiqueta
while IFS='|' read -r url titulo idioma tags; do
    # Limpiar y convertir a minúsculas
    url=$(echo "$url" | xargs)
    titulo=$(echo "$titulo" | xargs)
    idioma=$(echo "$idioma" | xargs | tr '[:upper:]' '[:lower:]')
    tags=$(echo "$tags" | xargs | tr '[:upper:]' '[:lower:]')
    
    # Extraer primera etiqueta para agrupación
    primera_etiqueta=$(echo "$tags" | cut -d',' -f1)
    
    # Capitalizar primera letra de la etiqueta
    primera_etiqueta=$(echo "$primera_etiqueta" | sed 's/^./\U&/')
    
    # Guardar en archivo temporal con etiqueta como primer campo
    echo "$primera_etiqueta|$url|$titulo|$idioma|$tags" >> "$TEMP_FILE"
    
done < "$ARCHIVO_TEMP"

# Ordenar por primera etiqueta
sort "$TEMP_FILE" -o "$TEMP_FILE"

# Generar cada página
for ((PAGINA=1; PAGINA<=TOTAL_PAGINAS; PAGINA++)); do
    if [ $PAGINA -eq 1 ]; then
        ARCHIVO_SALIDA="index.gmi"
    else
        ARCHIVO_SALIDA="index-${PAGINA}.gmi"
    fi
    
    # Calcular rangos para la página actual
    INICIO=$(( (PAGINA - 1) * ENLACES_POR_PAGINA + 1 ))
    FIN=$(( PAGINA * ENLACES_POR_PAGINA ))
    if [ $FIN -gt $TOTAL_UNICOS ]; then
        FIN=$TOTAL_UNICOS
    fi
    ENLACES_EN_PAGINA=$(( FIN - INICIO + 1 ))
    
    # Crear el archivo de la página
    echo "# Chimaque 🌀🌴" > "$ARCHIVO_SALIDA"
    echo "" >> "$ARCHIVO_SALIDA"
    echo "\`\`\`" >> "$ARCHIVO_SALIDA"
    echo "Total Enlaces = $TOTAL_UNICOS" >> "$ARCHIVO_SALIDA"
    if [ $TOTAL_PAGINAS -gt 1 ]; then
        echo "Página $PAGINA de $TOTAL_PAGINAS ($ENLACES_EN_PAGINA enlaces)" >> "$ARCHIVO_SALIDA"
    fi
    echo "\`\`\`" >> "$ARCHIVO_SALIDA"
    echo "" >> "$ARCHIVO_SALIDA"
    
    # Navegación entre páginas (solo si hay más de 1 página)
    if [ $TOTAL_PAGINAS -gt 1 ]; then
        echo "Ir a página:" >> "$ARCHIVO_SALIDA"
        for ((p=1; p<=TOTAL_PAGINAS; p++)); do
            if [ $p -eq 1 ]; then
                ARCHIVO_PAGINA="index.gmi"
            else
                ARCHIVO_PAGINA="index-${p}.gmi"
            fi
            if [ $p -eq $PAGINA ]; then
                echo "[$p] " >> "$ARCHIVO_SALIDA"
            else
                echo "=> $ARCHIVO_PAGINA $p" >> "$ARCHIVO_SALIDA"
            fi
        done
        echo "" >> "$ARCHIVO_SALIDA"
    fi
    
    echo "=> etiquetas.gmi 🏷️ Explorar por etiquetas" >> "$ARCHIVO_SALIDA"
    echo "" >> "$ARCHIVO_SALIDA"
    echo "## Enlaces" >> "$ARCHIVO_SALIDA"
    echo "" >> "$ARCHIVO_SALIDA"
    
    # Variable para llevar control de la última etiqueta
    ultima_etiqueta=""
    contador_enlaces=0
    primera_linea_grupo=true
    
    # Leer del archivo temporal ordenado y procesar solo los enlaces de esta página
    while IFS='|' read -r primera_etiqueta url titulo idioma tags && [ $contador_enlaces -lt $FIN ]; do
        contador_enlaces=$((contador_enlaces + 1))
        
        # Saltar enlaces de páginas anteriores
        if [ $contador_enlaces -lt $INICIO ]; then
            continue
        fi
        
        # Si cambió la etiqueta, crear nuevo encabezado
        if [ "$primera_etiqueta" != "$ultima_etiqueta" ]; then
            # No añadir línea vacía antes del primer grupo
            if [ "$ultima_etiqueta" != "" ]; then
                echo "" >> "$ARCHIVO_SALIDA"
            fi
            echo "### $primera_etiqueta" >> "$ARCHIVO_SALIDA"
            echo "" >> "$ARCHIVO_SALIDA"
            ultima_etiqueta="$primera_etiqueta"
            primera_linea_grupo=true
        fi
        
        # Mostrar solo título e idioma (sin etiquetas)
        echo "=> $url $titulo [$idioma]" >> "$ARCHIVO_SALIDA"
        primera_linea_grupo=false
        
    done < "$TEMP_FILE"
    
    echo "" >> "$ARCHIVO_SALIDA"
    echo "---" >> "$ARCHIVO_SALIDA"
    
    # Navegación al final también
    if [ $TOTAL_PAGINAS -gt 1 ]; then
        echo "" >> "$ARCHIVO_SALIDA"
        echo "Ir a página:" >> "$ARCHIVO_SALIDA"
        for ((p=1; p<=TOTAL_PAGINAS; p++)); do
            if [ $p -eq 1 ]; then
                ARCHIVO_PAGINA="index.gmi"
            else
                ARCHIVO_PAGINA="index-${p}.gmi"
            fi
            if [ $p -eq $PAGINA ]; then
                echo "[$p] " >> "$ARCHIVO_SALIDA"
            else
                echo "=> $ARCHIVO_PAGINA $p" >> "$ARCHIVO_SALIDA"
            fi
        done
        echo "" >> "$ARCHIVO_SALIDA"
    fi
    
    # FOOTER CON VERSIÓN Y FECHA ← AÑADIDO
    echo "Chimaque 🌀🌴 versión $VERSION." >> "$ARCHIVO_SALIDA"
    echo "📊 Última actualización de datos: $FECHA_ACTUAL" >> "$ARCHIVO_SALIDA"
    
    echo "✅ $ARCHIVO_SALIDA generado con $ENLACES_EN_PAGINA entradas"
done

# Limpiar archivos temporales
rm "$TEMP_FILE" "$ARCHIVO_TEMP" "$ARCHIVO_UNICO"

echo "🎉 Procesamiento completado:"
echo "   - Total enlaces originales: $TOTAL_ORIGINAL"
echo "   - Total enlaces únicos: $TOTAL_UNICOS"
echo "   - Enlaces duplicados ignorados: $DUPLICADOS"
echo "   - Páginas generadas: $TOTAL_PAGINAS"
echo "   - Versión: $VERSION"
echo "   - Fecha: $FECHA_ACTUAL"
