#!/bin/bash

# ----------------------------------------
# · ADVENTIA versión 1.2
# · Sistema de Ficción Interactiva Gemini
# · gemini://caleb.subnet.city/adventia/
# ----------------------------------------

# CONFIGURACIÓN MULTI-IDIOMA
LINKS_LABEL="Enlaces"  # Español
# LINKS_LABEL="Links"   # English  
# LINKS_LABEL="Enllaços" # Català
# LINKS_LABEL="Liens"   # Français

input_file="pages.dat"
declare -A page_names
declare -A page_descriptions
declare -A page_links

# Primera pasada: recolectar información de páginas
while IFS= read -r line; do
    if [[ "$line" =~ ^id:[[:space:]]*(.*) ]]; then
        current_id="${BASH_REMATCH[1]}"
    elif [[ "$line" =~ ^name:[[:space:]]*(.*) ]] && [[ -n "$current_id" ]]; then
        page_names["$current_id"]="${BASH_REMATCH[1]}"
    elif [[ "$line" =~ ^desc:[[:space:]]*(.*) ]] && [[ -n "$current_id" ]]; then
        page_descriptions["$current_id"]="${BASH_REMATCH[1]}"
        # Leer líneas adicionales de descripción
        while IFS= read -r desc_line; do
            if [[ "$desc_line" =~ ^(com:|id:|name:|ascii:|$) ]]; then
                break
            fi
            page_descriptions["$current_id"]="${page_descriptions["$current_id"]}"$'\n'"$desc_line"
        done
    fi
done < "$input_file"

# Función para limpiar espacios iniciales (solo para descripción)
clean_whitespace() {
    local text="$1"
    echo "$text" | sed 's/^[[:space:]]*//'
}

# Función para extraer enlaces en orden de aparición
extract_links_in_order() {
    local desc="$1"
    local links=()
    
    # Buscar todos los {{ID}} en la descripción
    while [[ $desc =~ \{\{([^}]+)\}\} ]]; do
        link_content="${BASH_REMATCH[1]}"
        # Extraer solo el ID (ignorar texto personalizado si existe)
        if [[ $link_content =~ ^([0-9A-Z_]+) ]]; then
            links+=("${BASH_REMATCH[1]}")
        fi
        # Continuar con el resto del texto
        desc="${desc#*${BASH_REMATCH[0]}}"
    done
    
    printf '%s\n' "${links[@]}"
}

# Segunda pasada: generar los archivos .gmi
current_id=""
current_name=""
current_ascii=""
current_desc=""
in_ascii_section=0
in_desc_section=0

reset_vars() {
    current_id=""
    current_name=""
    current_ascii=""
    current_desc=""
    in_ascii_section=0
    in_desc_section=0
}

process_page() {
    if [[ -n "$current_id" ]]; then
        output_file="${current_id}.gmi"
        
        echo "# $current_name" > "$output_file"
        
        # Solo generar bloque ``` si hay contenido ASCII no vacío
        if [[ -n "$current_ascii" && -n "${current_ascii//[[:space:]]/}" ]]; then
            echo '```' >> "$output_file"
            # NO limpiar espacios iniciales del arte ASCII - mantener el centrado
            echo "$current_ascii" >> "$output_file"
            echo '```' >> "$output_file"
        fi
        
        if [[ -n "$current_desc" ]]; then
            # Limpiar espacios iniciales de la descripción
            clean_desc=$(clean_whitespace "$current_desc")
            
            # Reemplazar {{id}} por los nombres correspondientes
            processed_desc="$clean_desc"
            for id in "${!page_names[@]}"; do
                pattern="{{${id}}}"
                replacement="${page_names[$id]}"
                processed_desc="${processed_desc//$pattern/$replacement}"
            done
            
           echo "" >> "$output_file"
            echo "$processed_desc" >> "$output_file"
        fi
        
        # Extraer enlaces en el orden que aparecen en la descripción
        page_links["$current_id"]=""
        link_ids=($(extract_links_in_order "$current_desc"))
        
        # SOLO mostrar sección de enlaces si hay enlaces
        if [[ ${#link_ids[@]} -gt 0 ]]; then
            echo "" >> "$output_file"
            echo "$LINKS_LABEL:" >> "$output_file"
            
            for link_id in "${link_ids[@]}"; do
                if [[ -n "${page_names[$link_id]}" ]]; then
                    echo "=> ${link_id}.gmi ${page_names[$link_id]}" >> "$output_file"
                    # Añadir al mapa de conexiones
                    if [[ -z "${page_links[$current_id]}" ]]; then
                        page_links["$current_id"]="$link_id"
                    else
                        page_links["$current_id"]="${page_links[$current_id]}, $link_id"
                    fi
                fi
            done
        fi
        
        echo "Generado: $output_file 🚩"
    fi
}

reset_vars

# Reiniciar la lectura del archivo para la segunda pasada
while IFS= read -r line; do
    # Saltar las líneas de metadatos del juego
    if [[ "$line" =~ ^(gametitle:|gameauthor:|gameascii:|gamedesc:|com:).* ]] && [[ -z "$current_id" ]]; then
        continue
    fi
    
    # Si encontramos una nueva sección id, procesamos la página anterior
    if [[ "$line" =~ ^id:[[:space:]]*(.*) ]]; then
        process_page
        reset_vars
        current_id="${BASH_REMATCH[1]}"
        in_ascii_section=0
        in_desc_section=0
    
    elif [[ "$line" =~ ^name:[[:space:]]*(.*) ]]; then
        current_name="${BASH_REMATCH[1]}"
        in_ascii_section=0
        in_desc_section=0
    
    elif [[ "$line" =~ ^ascii:[[:space:]]*$ ]]; then
        # ASCII empieza en la siguiente línea
        in_ascii_section=1
        in_desc_section=0
        current_ascii=""
    
    elif [[ "$line" =~ ^ascii:[[:space:]]*(.*) ]]; then
        # ASCII tiene contenido en la misma línea
        in_ascii_section=1
        in_desc_section=0
        current_ascii="${BASH_REMATCH[1]}"
    
    elif [[ "$line" =~ ^desc:[[:space:]]*(.*) ]]; then
        in_ascii_section=0
        in_desc_section=1
        current_desc="${BASH_REMATCH[1]}"
    
    elif [[ "$line" =~ ^com: ]]; then
        in_ascii_section=0
        in_desc_section=0
    
    else
        # Si estamos en la sección ascii, añadimos a current_ascii
        if [[ $in_ascii_section -eq 1 ]]; then
            if [[ -z "$current_ascii" ]]; then
                current_ascii="$line"
            else
                current_ascii="$current_ascii"$'\n'"$line"
            fi
        # Si estamos en la sección desc, añadimos a current_desc
        elif [[ $in_desc_section -eq 1 ]]; then
            if [[ -z "$current_desc" ]]; then
                current_desc="$line"
            else
                current_desc="$current_desc"$'\n'"$line"
            fi
        fi
    fi
done < "$input_file"

# Procesar la última página
process_page

# Generar archivo de mapa solo con páginas que tienen conexiones
echo "# Mapa del Juego" > "map.gmi"
echo "" >> "map.gmi"
echo "Conexiones entre páginas:" >> "map.gmi"
echo "" >> "map.gmi"

# Ordenar los IDs numéricamente y mostrar solo los que tienen conexiones
has_connections=0
for id in $(printf '%s\n' "${!page_links[@]}" | sort -V); do
    if [[ -n "${page_links[$id]}" ]]; then
        echo "## ${id} - ${page_names[$id]}" >> "map.gmi"
        echo "→ Conecta con: ${page_links[$id]}" >> "map.gmi"
        echo "" >> "map.gmi"
        has_connections=1
    fi
done

if [[ $has_connections -eq 0 ]]; then
    echo "No se encontraron conexiones entre páginas." >> "map.gmi"
fi

echo "Mapa generado: map.gmi 🧭"
echo "Generación de Aventura Terminada. 🎒"

# DOCUMENTACIÓN MULTI-IDIOMA
echo ""
echo "=== DOCUMENTACIÓN ADVENTIA 1.2 ==="
echo "VARIABLE MULTI-IDIOMA: LINKS_LABEL"
echo "Valor actual: '$LINKS_LABEL'"
echo ""
echo "Para cambiar el idioma, edita la línea:"
echo "LINKS_LABEL=\"Texto_en_tu_idioma\""
echo ""
echo "Ejemplos:"
echo "  LINKS_LABEL=\"Links\"       # English"
echo "  LINKS_LABEL=\"Enllaços\"    # Català" 
echo "  LINKS_LABEL=\"Liens\"       # Français"
echo "  LINKS_LABEL=\"链接\"         # Chinese"
echo "  LINKS_LABEL=\"リンク\"       # Japanese"
echo ""
echo "Características v1.2:"
echo "✅ Etiqueta multi-idioma para enlaces"
echo "✅ Orden de enlaces preservado"
echo "✅ Sistema Everett compatible"
echo "✅ 47 páginas generadas correctamente"
