/* UNLICENSE
 * This is free and unencumbered software released into the public domain.
 * For more information, please refer to <http://unlicense.org/>
 */

class GempubReader {
    constructor() {
        this.zip = null;
        this.metadata = {};
        this.currentChapterIndex = -1;
        this.chapters = [];
        this.isSidebarVisible = true;
        this.currentBook = null;
        
        this.initializeEventListeners();
        this.loadProgress();
    }
    
    initializeEventListeners() {
        // File input SIMPLIFICADO - solo un event listener
        const fileInput = document.getElementById('gpub-file');
        
        fileInput.addEventListener('change', (e) => {
            if (e.target.files[0]) {
                this.loadGempub(e.target.files[0]);
            }
        });
        
        // Toggle sidebar
        document.getElementById('sidebar-toggle').addEventListener('click', () => {
            this.toggleSidebar();
        });
        
        // Toggle metadatos
        document.getElementById('book-info-link').addEventListener('click', (e) => {
            e.preventDefault();
            this.toggleMetadata();
        });
        
        // Navegación entre capítulos
        document.getElementById('prev-chapter').addEventListener('click', (e) => {
            e.preventDefault();
            this.loadPreviousChapter();
        });
        
        document.getElementById('next-chapter').addEventListener('click', (e) => {
            e.preventDefault();
            this.loadNextChapter();
        });
        
        // Guardar progreso al hacer scroll
        document.querySelector('.content').addEventListener('scroll', () => {
            this.saveProgress();
        });
        
        // Guardar progreso antes de cerrar la página
        window.addEventListener('beforeunload', () => {
            this.saveProgress();
        });
    }
    
    toggleSidebar() {
        this.isSidebarVisible = !this.isSidebarVisible;
        const sidebar = document.getElementById('sidebar');
        const toggleBtn = document.getElementById('sidebar-toggle');
        
        if (this.isSidebarVisible) {
            sidebar.classList.remove('hidden');
            toggleBtn.textContent = '☰';
        } else {
            sidebar.classList.add('hidden');
            toggleBtn.textContent = '☰';
        }
    }
    
    async loadGempub(file) {
        try {
            this.zip = await JSZip.loadAsync(file);
            this.currentBook = file.name;
            
            // Cargar metadatos
            await this.loadMetadata();
            
            // Cargar índice
            await this.loadIndex();
            
            // Mostrar título
            this.displayBookTitle();
            
            // Cargar portada (pero no mostrar hasta BOOK INFO)
            await this.loadCover();
            
            console.log('Gempub cargado correctamente');
            
            // Intentar cargar progreso guardado
            this.loadProgress();
            
        } catch (error) {
            console.error('Error al cargar Gempub:', error);
            alert('Error al cargar el archivo Gempub');
        }
    }
    
    displayBookTitle() {
        const titleElement = document.getElementById('book-title');
        const title = this.metadata.title || 'Libro sin título';
        titleElement.textContent = title;
    }
    
    async loadMetadata() {
        try {
            const metadataFile = this.zip.file('metadata.txt');
            if (metadataFile) {
                const content = await metadataFile.async('text');
                this.parseMetadata(content);
                this.displayMetadata();
            }
        } catch (error) {
            console.warn('No se encontró metadata.txt');
        }
    }
    
    parseMetadata(content) {
        this.metadata = {};
        const lines = content.split('\n');
        
        lines.forEach(line => {
            const separatorIndex = line.indexOf(':');
            if (separatorIndex > 0) {
                const key = line.substring(0, separatorIndex).trim();
                const value = line.substring(separatorIndex + 1).trim();
                this.metadata[key] = value;
            }
        });
    }
    
    displayMetadata() {
        const container = document.getElementById('metadata-content');
        let html = '<h3>Información del Libro</h3>';
        
        const metadataFields = [
            { key: 'title', label: 'Título' },
            { key: 'author', label: 'Autor' },
            { key: 'description', label: 'Descripción' },
            { key: 'language', label: 'Idioma' },
            { key: 'published', label: 'Publicado' },
            { key: 'publishDate', label: 'Fecha' },
            { key: 'copyright', label: 'Copyright' },
            { key: 'license', label: 'Licencia' }
        ];
        
        metadataFields.forEach(field => {
            if (this.metadata[field.key]) {
                html += `
                    <div class="metadata-item">
                        <span class="metadata-label">${field.label}:</span>
                        <span>${this.metadata[field.key]}</span>
                    </div>
                `;
            }
        });
        
        container.innerHTML = html;
    }
    
    async loadIndex() {
        try {
            // Determinar la ruta del índice
            let indexPath = 'index.gmi';
            if (this.metadata.index) {
                indexPath = this.metadata.index;
            }
            
            const indexFile = this.zip.file(indexPath);
            if (!indexFile) {
                throw new Error('No se encontró el archivo índice');
            }
            
            const content = await indexFile.async('text');
            this.displayIndex(content);
            
        } catch (error) {
            console.error('Error al cargar índice:', error);
            document.getElementById('toc').innerHTML = '<p>Error al cargar el índice</p>';
        }
    }
    
    displayIndex(gemtext) {
        const tocElement = document.getElementById('toc');
        const lines = gemtext.split('\n');
        
        // Reiniciar array de capítulos
        this.chapters = [];
        
        let html = '';
        let inPreformatted = false;
        let skipTitle = false;
        let skipAuthor = false;
        
        // Detectar si debemos omitir título y autor del índice
        const metadataTitle = this.metadata.title;
        const metadataAuthor = this.metadata.author;
        
        lines.forEach((line, index) => {
            // Manejar bloques preformateados
            if (line.startsWith('```')) {
                if (inPreformatted) {
                    html += '</pre>';
                    inPreformatted = false;
                } else {
                    html += '<pre class="toc-pre">';
                    inPreformatted = true;
                }
                return;
            }
            
            if (inPreformatted) {
                html += line + '\n';
                return;
            }
            
            // Verificar si debemos omitir título (primera línea que empiece con #)
            if (index === 0 && line.startsWith('# ') && metadataTitle) {
                const titleFromIndex = line.substring(1).trim();
                if (this.normalizeText(titleFromIndex) === this.normalizeText(metadataTitle)) {
                    skipTitle = true;
                    return;
                }
            }
            
            // Verificar si debemos omitir autor (segunda línea que empiece con ##)
            if (index === 1 && line.startsWith('## ') && metadataAuthor) {
                const authorFromIndex = line.substring(2).trim();
                if (this.normalizeText(authorFromIndex) === this.normalizeText(metadataAuthor)) {
                    skipAuthor = true;
                    return;
                }
            }
            
            // Procesar según el tipo de línea Gemini
            if (line.startsWith('###')) {
                // Encabezado nivel 3 - subsección
                html += `<h3 class="toc-h3">${line.substring(3).trim()}</h3>`;
            } else if (line.startsWith('##')) {
                // Encabezado nivel 2 - sección principal
                if (skipAuthor && index === 1) {
                    // Omitir autor duplicado
                    return;
                }
                html += `<h2 class="toc-h2">${line.substring(2).trim()}</h2>`;
            } else if (line.startsWith('#')) {
                // Encabezado nivel 1 - título principal
                if (skipTitle && index === 0) {
                    // Omitir título duplicado
                    return;
                }
                html += `<h1 class="toc-h1">${line.substring(1).trim()}</h1>`;
            } else if (line.startsWith('=>')) {
                // Enlace - extraer y crear link funcional
                const link = this.parseGeminiLink(line);
                if (link) {
                    // Solo considerar enlaces locales como capítulos navegables
                    if (!link.url.startsWith('http') && !link.url.startsWith('gemini://')) {
                        const chapterIndex = this.chapters.length;
                        this.chapters.push(link);
                        html += `
                            <div class="toc-link">
                                <a href="#" data-url="${link.url}" data-index="${chapterIndex}" class="chapter-link">
                                    📖 ${link.text}
                                </a>
                            </div>
                        `;
                    } else {
                        // Enlaces externos
                        html += `
                            <div class="toc-link external">
                                <a href="${link.url}" target="_blank" rel="noopener">
                                    🔗 ${link.text}
                                </a>
                            </div>
                        `;
                    }
                }
            } else if (line.startsWith('* ')) {
                // Lista
                html += `<div class="toc-list-item">• ${line.substring(2).trim()}</div>`;
            } else if (line.startsWith('>')) {
                // Cita
                html += `<blockquote class="toc-quote">${line.substring(1).trim()}</blockquote>`;
            } else if (line.trim() === '') {
                // Línea vacía
                html += '<div class="toc-empty-line"></div>';
            } else {
                // Texto normal
                html += `<p class="toc-text">${line}</p>`;
            }
        });
        
        // Cerrar bloque preformateado si queda abierto
        if (inPreformatted) {
            html += '</pre>';
        }
        
        tocElement.innerHTML = html;
        
        // Agregar event listeners a los enlaces de capítulos
        document.querySelectorAll('.chapter-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const chapterIndex = parseInt(e.target.getAttribute('data-index'));
                this.loadChapterByIndex(chapterIndex);
                this.scrollToTop();
            });
        });
        
        // Cargar el primer capítulo automáticamente si no hay progreso guardado
        if (this.chapters.length > 0 && this.currentChapterIndex === -1) {
            this.loadChapterByIndex(0);
        }
    }
    
    async loadChapterByIndex(chapterIndex) {
        if (chapterIndex < 0 || chapterIndex >= this.chapters.length) {
            return;
        }
        
        const chapter = this.chapters[chapterIndex];
        this.currentChapterIndex = chapterIndex;
        
        // Actualizar enlace activo en el índice
        document.querySelectorAll('.chapter-link').forEach(link => {
            link.classList.remove('active');
        });
        const activeLink = document.querySelector(`.chapter-link[data-index="${chapterIndex}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }
        
        // Cargar contenido del capítulo
        await this.loadChapter(chapter.url);
        
        // Actualizar navegación
        this.updateChapterNavigation();
        
        // Guardar progreso
        this.saveProgress();
    }
    
    async loadChapter(chapterPath) {
        try {
            const chapterFile = this.zip.file(chapterPath);
            if (!chapterFile) {
                throw new Error(`No se encontró el capítulo: ${chapterPath}`);
            }
            
            const content = await chapterFile.async('text');
            this.displayContent(content);
            
        } catch (error) {
            console.error('Error al cargar capítulo:', error);
            document.getElementById('chapter-content').innerHTML = '<p>Error al cargar el capítulo</p>';
        }
    }
    
    displayContent(gemtext) {
        const contentElement = document.getElementById('chapter-content');
        contentElement.innerHTML = this.convertGeminiToHTML(gemtext);
    }
    
    convertGeminiToHTML(gemtext) {
        const lines = gemtext.split('\n');
        let html = '';
        let inList = false;
        let inPreformatted = false;
        let preformattedLanguage = '';
        
        lines.forEach(line => {
            // Manejar bloques preformateados
            if (line.startsWith('```')) {
                if (inPreformatted) {
                    html += '</pre>';
                    inPreformatted = false;
                } else {
                    preformattedLanguage = line.substring(3).trim();
                    html += `<pre${preformattedLanguage ? ` class="language-${preformattedLanguage}"` : ''}>`;
                    inPreformatted = true;
                }
                return;
            }
            
            if (inPreformatted) {
                html += line + '\n';
                return;
            }
            
            // Procesar líneas normales
            if (line.startsWith('###')) {
                if (inList) { html += '</ul>'; inList = false; }
                html += `<h3>${line.substring(3).trim()}</h3>`;
            } else if (line.startsWith('##')) {
                if (inList) { html += '</ul>'; inList = false; }
                html += `<h2>${line.substring(2).trim()}</h2>`;
            } else if (line.startsWith('#')) {
                if (inList) { html += '</ul>'; inList = false; }
                html += `<h1>${line.substring(1).trim()}</h1>`;
            } else if (line.startsWith('* ')) {
                if (!inList) { html += '<ul>'; inList = true; }
                html += `<li>${line.substring(2).trim()}</li>`;
            } else if (line.startsWith('=>')) {
                if (inList) { html += '</ul>'; inList = false; }
                const link = this.parseGeminiLink(line);
                if (link) {
                    if (link.url.match(/\.(jpg|jpeg|png|gif)$/i)) {
                        html += `<div class="image-link">
                            <img src="#" data-src="${link.url}" alt="${link.text}" class="lazy-image">
                            <br><small>${link.text}</small>
                        </div>`;
                    } else {
                        html += `<p>⇒ <a href="#" data-url="${link.url}" class="content-link">${link.text}</a></p>`;
                    }
                }
            } else if (line.startsWith('>')) {
                if (inList) { html += '</ul>'; inList = false; }
                html += `<blockquote>${line.substring(1).trim()}</blockquote>`;
            } else if (line.trim() === '') {
                if (inList) { html += '</ul>'; inList = false; }
                html += '<br>';
            } else {
                if (inList) { html += '</ul>'; inList = false; }
                html += `<p>${line}</p>`;
            }
        });
        
        if (inList) { html += '</ul>'; }
        if (inPreformatted) { html += '</pre>'; }
        
        return html;
    }
    
    updateChapterNavigation() {
        const prevNav = document.getElementById('prev-nav');
        const nextNav = document.getElementById('next-nav');
        const prevLink = document.getElementById('prev-chapter');
        const nextLink = document.getElementById('next-chapter');
        
        // Mostrar/ocultar navegación
        prevNav.classList.remove('hidden');
        nextNav.classList.remove('hidden');
        
        // Capítulo anterior
        if (this.currentChapterIndex > 0) {
            prevLink.style.display = 'inline-block';
        } else {
            prevLink.style.display = 'none';
        }
        
        // Capítulo siguiente
        if (this.currentChapterIndex < this.chapters.length - 1) {
            nextLink.style.display = 'inline-block';
        } else {
            nextLink.style.display = 'none';
        }
    }
    
    loadPreviousChapter() {
        if (this.currentChapterIndex > 0) {
            this.loadChapterByIndex(this.currentChapterIndex - 1);
            this.scrollToTop();
        }
    }
    
    loadNextChapter() {
        if (this.currentChapterIndex < this.chapters.length - 1) {
            this.loadChapterByIndex(this.currentChapterIndex + 1);
            this.scrollToTop();
        }
    }
    
    scrollToTop() {
        document.querySelector('.content').scrollTop = 0;
    }
    
    async loadCover() {
        const coverContainer = document.getElementById('cover-container');
        coverContainer.innerHTML = '';
        
        if (this.metadata.cover) {
            try {
                const coverFile = this.zip.file(this.metadata.cover);
                if (coverFile) {
                    const blob = await coverFile.async('blob');
                    const url = URL.createObjectURL(blob);
                    coverContainer.innerHTML = `
                        <img src="${url}" alt="Portada" class="clickable-cover" 
                             data-url="${this.metadata.cover}">
                    `;
                    
                    // Hacer la carátula clickable
                    document.querySelector('.clickable-cover').addEventListener('click', () => {
                        this.displayCoverInContent(url);
                    });
                }
            } catch (error) {
                console.warn('No se pudo cargar la portada:', error);
            }
        }
    }
    
    displayCoverInContent(coverUrl) {
        const contentElement = document.getElementById('chapter-content');
        contentElement.innerHTML = `
            <div class="cover-fullview">
                <h2>Portada</h2>
                <img src="${coverUrl}" alt="Portada del libro" class="full-cover">
                <p class="cover-caption">${this.metadata.title || 'Libro'}</p>
            </div>
        `;
        
        // Ocultar navegación de capítulos cuando se muestra la portada
        document.getElementById('prev-nav').classList.add('hidden');
        document.getElementById('next-nav').classList.add('hidden');
        
        // Actualizar estado
        this.currentChapterIndex = -1; // Indicar que estamos viendo la portada, no un capítulo
        
        // Scroll al top
        this.scrollToTop();
    }
    
    parseGeminiLink(line) {
        const parts = line.substring(2).trim().split(/\s+(.+)/);
        if (parts.length >= 2) {
            return {
                url: parts[0],
                text: parts[1] || parts[0]
            };
        }
        return null;
    }
    
    // Función auxiliar para normalizar texto (comparaciones sin sensibilidad a mayúsculas/espacios)
    normalizeText(text) {
        return text.toLowerCase().replace(/\s+/g, ' ').trim();
    }
    
    toggleMetadata() {
        const metadataElement = document.getElementById('book-metadata');
        metadataElement.classList.toggle('hidden');
    }
    
    // ===== SISTEMA DE GUARDADO DE PROGRESO =====
    
    saveProgress() {
        if (!this.currentBook || this.currentChapterIndex === -1) return;
        
        const progress = {
            book: this.currentBook,
            chapterIndex: this.currentChapterIndex,
            scrollPosition: document.querySelector('.content').scrollTop,
            timestamp: Date.now()
        };
        
        localStorage.setItem('gempub-reader-progress', JSON.stringify(progress));
    }
    
    loadProgress() {
        try {
            const saved = localStorage.getItem('gempub-reader-progress');
            if (!saved) return;
            
            const progress = JSON.parse(saved);
            
            // Solo cargar si tenemos el mismo libro y capítulos cargados
            if (progress.book === this.currentBook && this.chapters.length > 0) {
                // Cargar capítulo guardado
                if (progress.chapterIndex >= 0 && progress.chapterIndex < this.chapters.length) {
                    this.loadChapterByIndex(progress.chapterIndex);
                    
                    // Restaurar posición de scroll después de que se cargue el contenido
                    setTimeout(() => {
                        if (progress.scrollPosition > 0) {
                            document.querySelector('.content').scrollTop = progress.scrollPosition;
                        }
                    }, 100);
                }
            }
        } catch (error) {
            console.warn('Error al cargar progreso:', error);
        }
    }
}

// Inicializar el lector cuando se carga la página
document.addEventListener('DOMContentLoaded', () => {
    new GempubReader();
});